import logging
import datetime

import luigi

from crypta.graph.v1.python.lib.luigi import yt_luigi
from crypta.graph.v1.python.rtcconf import config
from crypta.graph.v1.python.utils import mr_utils as mr
from crypta.graph.v1.python.utils import yt_clients
from crypta.graph.v1.python.utils.utils import send_reactor_event


class GraphHistorySnapshot(yt_luigi.BaseYtTask, yt_clients.YtClientMixin):

    date = luigi.Parameter()
    parent_task = luigi.Parameter()
    tags = ["v1"]

    def requires(self):
        if self.parent_task:
            return self.parent_task
        else:
            return []

    def __init__(self, date, parent_task):
        # every day
        day_tables = {config.YT_OUTPUT_FOLDER + date + "/exact/cluster/vertices": "vertices"}

        # every month
        month_tables = {
            config.GRAPH_YT_DICTS_FOLDER + "yuid_with_all": "yuid_with_all",
            config.GRAPH_YT_DICTS_FOLDER + "yuid_with_all_bad": "yuid_with_all_bad",
            config.GRAPH_YT_DICTS_FOLDER + "dev_info_yt": "dev_info_yt",
            config.YT_OUTPUT_FOLDER + date + "/exact/cluster/edges": "edges",
        }

        self.copy_today = day_tables
        self.dt = datetime.datetime.strptime(date, "%Y-%m-%d")
        if self.dt.day == 1:  # every first day of month
            self.copy_today.update(month_tables)

        super(GraphHistorySnapshot, self).__init__(date=date, parent_task=parent_task)

    def run(self):
        logging.info("Preparing history snapshot for %s" % self.date)

        history_f = self.out_f("history")
        mr.mkdir(history_f)

        for copy_from, copy_to in self.copy_today.iteritems():
            copy_to = history_f + copy_to

            if self.yt.exists(copy_from):
                logging.info("Copying %s to %s" % (copy_from, copy_to))
                mr.copy(copy_from, copy_to)
            else:
                logging.warn("%s doesn't exist" % copy_from)

    def after_run(self):
        history_f = self.out_f("history")
        for _, target in self.copy_today.iteritems():
            send_reactor_event(self.dt, history_f + target)

    def output(self):
        history_f = self.out_f("history")
        return [
            yt_luigi.YtTarget(history_f + target_table, allow_empty=True)
            for _, target_table in self.copy_today.iteritems()
        ]

    def output_folders(self):
        return {"history": config.V1_ARCHIVE_FOLDER + self.date + "/"}
