# -*- coding: utf-8 -*-

import logging

import luigi

from crypta.graph.v1.python.data_imports.import_logs.graph_watch_log import ImportWatchLogDayTask
from crypta.graph.v1.python.infra.radius.radius_splunk_client import import_radius_log_to_local_file
from crypta.graph.v1.python.lib.luigi import base_luigi_task
from crypta.graph.v1.python.lib.luigi import yt_luigi
from crypta.graph.v1.python.rtcconf import config
from crypta.graph.v1.python.utils import mr_utils as mr
from crypta.graph.v1.python.utils import yt_clients
from crypta.graph.v1.python.utils.yql_utils import run_yql

logger = logging.getLogger()


class TvmConfig(object):
    def __init__(self, **kwargs):
        self.__dict__.update(**kwargs)


def read_password():
    pass_path = config.RADIUS_SPLUNK_PASS_PATH

    if not pass_path:
        if config.RADIUS_SPLUNK_PASSWORD is not None:
            return config.RADIUS_SPLUNK_PASSWORD
        raise Exception("RADIUS_SPLUNK_PASS_PATH is not set")

    with open(pass_path) as f:
        password_line = f.readline()
        if password_line:
            return password_line.strip()
        else:
            raise Exception("%s is empty" % pass_path)


def convert_timestamp(rec):
    rec["timestamp"] = long(rec["timestamp"])
    yield rec


def distinct_ip(key, recs):
    yield {"ip": key["ip"]}


def calculate_ips(yt_folder, date, store_days):
    radius_log_table = mr.get_date_table(yt_folder, date, "radius_log")
    ip_table = mr.get_date_table(yt_folder, date, "ips")

    yt_client = yt_clients.get_yt_client()
    yt_client.run_map_reduce(None, distinct_ip, radius_log_table, ip_table, reduce_by="ip")
    yt_client.run_sort(ip_table, ip_table, sort_by=["ip", "timestamp"])

    # distinct all ips for previous days
    ips_tables_range = mr.get_existing_date_tables(yt_folder, "ips", store_days)
    all_ips_table = mr.get_date_table(yt_folder, date, "all_radius_ips")

    yt_client.run_reduce(distinct_ip, ips_tables_range, all_ips_table, reduce_by="ip")
    yt_client.run_sort(all_ips_table, sort_by=["ip", "timestamp"])


def upload_results_to_yt(local_folder, yt_folder, date):
    logger.info("Uploading result to YT...")
    mr.mkdir(yt_folder + date)

    local_file = local_folder + date
    yt_table = mr.get_date_table(yt_folder, date, "radius_log")

    f = open(local_file)
    yt_client = yt_clients.get_yt_client()
    yt_client.write_table(yt_table, f, format="dsv", raw=True)
    yt_client.run_map(convert_timestamp, yt_table, yt_table)


class ImportRadiusSplunkLog(base_luigi_task.BaseTask):
    date = luigi.Parameter()
    priority = 1

    def run(self):
        splunk_url = config.RADIUS_SPLUNK_URL
        splunk_port = str(config.RADIUS_SPLUNK_PORT)
        splunk_user = config.RADIUS_SPLUNK_USR
        splunk_pass = read_password()

        local_path = config.RADIUS_LOG_LOCAL_FOLDER
        yt_path = config.RADIUS_LOG_YT_FOLDER

        if not config.RADIUS_TVM_SECRET:
            raise Exception("Env RADIUS_TVM_SECRET must be set")

        tvm_config = TvmConfig(
            client_id=config.RADIUS_CLIENT_TVM_ID,
            server_id=config.RADIUS_SERVER_TVM_ID,
            secret=config.RADIUS_TVM_SECRET,
        )

        import_radius_log_to_local_file(
            self.date, local_path, splunk_url, splunk_port, splunk_user, splunk_pass, tvm_config
        )
        upload_results_to_yt(local_path, yt_path, self.date)
        calculate_ips(yt_path, self.date, int(config.STORE_DAYS))

    def output(self):
        out_f = config.RADIUS_LOG_YT_FOLDER + self.date + "/"
        return [yt_luigi.YtTarget(out_f + "radius_log"), yt_luigi.YtTarget(out_f + "all_radius_ips")]


class FilterByRadius(yt_luigi.BaseYtTask):
    date = luigi.Parameter()

    def requires(self):
        return [ImportRadiusSplunkLog(self.date), ImportWatchLogDayTask(date=self.date, run_date=self.date)]

    def run(self):
        run_yql("RadiusFilter", dict(date=self.date))

    def output_folders(self):
        return {"yt_output": config.YT_OUTPUT_FOLDER + self.date}

    def output(self):
        return map(yt_luigi.YtTarget, (self.out_f("yt_output") + "/raw_links/watch_log_filtered_by_radius",))
