import contextlib
import functools
import os
import time
from datetime import datetime

import luigi
import re

from crypta.graph.v1.python.lib.crypta_api import report_task_status_to_api
from crypta.graph.v1.python.lib.luigi import base_luigi_task
from crypta.graph.v1.python.rtcconf import config
from crypta.graph.v1.python.utils import mr_utils as mr
from crypta.graph.v1.python.utils import yt_clients


REDATE = re.compile(r"\/\d{4}-\d{2}-\d{2}")


class YtTarget(luigi.Target, yt_clients.YtClientMixin):
    def __init__(self, table, allow_empty=False, crypta_api_notify=False):
        self.table = table
        self.allow_empty = allow_empty
        self.crypta_api_notify = crypta_api_notify
        self.unified_table_path = REDATE.sub("/day", self.table)
        self.param_kwargs = dict(
            table=table,
            allow_empty=allow_empty,
            crypta_api_notify=crypta_api_notify,
        )

    def exists(self):
        if self.yt.exists(self.table):
            if self.crypta_api_notify:
                report_task_status_to_api(self, "SUCCESS")
            return (
                self.allow_empty
                or (mr.row_count(self.table, yt_client=self.yt) > 0)
                or getattr(config, "YT_TARGET_EMPTY_TABLE_OK", False)
            )
        else:
            if self.crypta_api_notify and (datetime.now().hour > config.WAKE_UP_TIME):
                report_task_status_to_api(self, "FAILURE")
            return False

    def get_task_family(self):
        return "YtTarget_{}".format(self.unified_table_path)


class YtFolderTarget(luigi.Target, yt_clients.YtClientMixin):
    def __init__(self, folder, allow_empty=False, absolute_path=False):
        self.folder = folder
        self.allow_empty = allow_empty
        self.absolute_path = absolute_path

    def exists(self):
        if not self.yt.exists(self.folder):
            return False
        if not self.allow_empty:
            return len(mr.ls(self.folder, absolute_path=self.absolute_path, yt_client=self.yt)) > 0
        else:
            return True


class YtDateTarget(luigi.Target, yt_clients.YtClientMixin):
    def __init__(self, table, date):
        self.table = table
        self.date = date

    def exists(self):
        if self.yt.exists(self.table):
            generate_date = mr.get_generate_date(self.table, yt_client=self.yt)
            return self.date <= generate_date
        else:
            return False


class YtDateAttributeTarget(luigi.Target, yt_clients.YtClientMixin):
    def __init__(self, path, attribute_name, date):
        self.path = path
        self.attribute_name = attribute_name
        self.date = date

    def exists(self):
        if self.yt.exists(self.path):
            dt = self.yt.get_attribute(self.path, self.attribute_name, "1970-01-01")
            return self.date <= dt

        return False

    def set_date(self, dt=None):
        if dt is None:
            dt = self.date
        self.yt.set_attribute(self.path, self.attribute_name, dt)


class YtDateColumnTarget(YtDateTarget, yt_clients.YtClientMixin):
    def __init__(self, table, column, date):
        super(YtDateColumnTarget, self).__init__(table, date)
        self.column = column

    def exists(self):
        table_exists = super(YtDateColumnTarget, self).exists()
        return table_exists and self.column in self.yt.read_table(self.table, format="json", raw=False).next()


class YtComboLogTarget(luigi.Target, yt_clients.YtClientMixin):
    def __init__(self, link_dir, date):
        self.link_dir = link_dir
        self.date = date

    def exists(self):
        return all([self.yt.exists(x) for x in self.get_tables()])

    def get_tables(self):
        return [os.path.join(x, self.date) for x in self.yt.list(self.link_dir, absolute=True)]


class FileTarget(luigi.Target, yt_clients.YtClientMixin):
    def __init__(self, path):
        self.path = path

    def exists(self):
        return self.yt.exists(self.path)


class FilePrepared(luigi.ExternalTask):
    yt_path = luigi.Parameter()

    def output(self):
        return FileTarget(self.yt_path)


class TodayFileTarget(luigi.Target, yt_clients.YtClientMixin):
    def __init__(self, filename, date):
        self.filename = filename + ".date"
        self.date = date

    @staticmethod
    def done(filename, date):
        filename += ".date"
        with open(filename, "w") as fw:
            fw.write(date)

    def exists(self):
        if not os.path.exists(self.filename):
            return False
        with open(self.filename, "r") as fr:
            existing_date = fr.readline().strip()
            return self.date <= existing_date


class YtAttrTarget(luigi.Target, yt_clients.YtClientMixin):
    def __init__(self, path, attr):
        self.path = path
        self.attr = attr

    def exists(self):
        return self.yt.exists(self.path) and bool(mr.safe_get_attribute(self.path, self.attr, yt_client=self.yt))

    def complete(self):
        self.yt.set_attribute(self.path, self.attr, True)


class ExternalFolder(luigi.ExternalTask):

    folder = luigi.Parameter()
    allow_empty = luigi.Parameter(default=False)
    absolute_path = luigi.Parameter(default=False)

    def output(self):
        return YtFolderTarget(self.folder, self.allow_empty, self.absolute_path)


class ExternalInput(luigi.ExternalTask):

    table = luigi.Parameter()
    allow_empty = luigi.Parameter(default=False)

    def output(self):
        return YtTarget(self.table, self.allow_empty, crypta_api_notify=True)


class ExternalComboLogInput(luigi.ExternalTask):
    link_dir = luigi.Parameter()
    date = luigi.Parameter()

    def output(self):
        return YtComboLogTarget(self.link_dir, self.date)


class ExternalInputLastDate(luigi.ExternalTask, yt_clients.YtClientMixin):

    folder = luigi.Parameter()

    def output(self):
        fail_count = 0
        last_error = None
        while fail_count < 10:
            try:
                table = mr.get_last_table(self.folder, yt_client=self.yt)
                return YtTarget(table)
            except Exception as e:
                fail_count += 1
                last_error = e
                time.sleep(5)

        raise last_error


class YesterdayDictInput(ExternalInput):
    def __init__(self, table_name):
        table = config.GRAPH_YT_DICTS_FOLDER + table_name
        super(YesterdayDictInput, self).__init__(table)


class PostGraphTask(base_luigi_task.BaseTask, yt_clients.YtClientMixin):
    date = luigi.Parameter()
    name = luigi.Parameter()

    def run(self):
        self.run_post_graph()
        set_current_time_attr(self.path(), self.name, self.yt)

    def run_post_graph(self):
        pass

    def output(self):
        return YtAttrTarget(self.path(), self.name)

    def path(self):
        return config.YT_OUTPUT_FOLDER + self.date


@contextlib.contextmanager
def nullcontext():
    yield


class BaseYtTask(base_luigi_task.BaseTask, yt_clients.YtClientMixin):
    with_transaction = True
    transaction_timeout = None

    def __init__(self, *args, **kwargs):
        self.run = self.transaction_wrap(self.run)

        super(BaseYtTask, self).__init__(*args, **kwargs)

    def transaction_wrap(self, run_func):
        @functools.wraps(run_func)
        def wrapped(*args, **kwargs):
            self.before_run()
            self.tx = (
                self.yt.Transaction(timeout=self.transaction_timeout, attributes={"title": self.task_id})
                if self.with_transaction
                else nullcontext()
            )
            with self.tx:
                run_func(*args, **kwargs)
            self.after_run()

        return wrapped

    def before_run(self):
        pass

    def after_run(self):
        pass

    def input_folders(self):
        raise NotImplementedError

    def workdir(self):
        raise Exception("No workdir set")

    def output_folders(self):
        raise NotImplementedError

    def in_f(self, folder_alias):
        """
        Get input folder with specific alias from input_folders mapping
        """
        return self.input_folders()[folder_alias]

    def out_f(self, folder_alias):
        """
        Get output folder with specific alias from output_folders mapping
        """
        return self.output_folders()[folder_alias]


def run_task(task_name, date, memory_limit=1024):
    luigi.run([task_name, "--date", date, "--workers", "10", "--local-scheduler"])


def set_current_time_attr(path, attr, yt_client):
    dt = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    yt_client.set_attribute(path, attr, dt)
