import datetime
import logging
import time

import requests
from requests.packages.urllib3 import Retry

success_statuses = ["SUCCESS"]

fail_statuses = ["STOPPED", "EXCEPTION", "TIMEOUT", "FAILURE", "DELETED"]


class SandboxApi(object):
    def __init__(self, oauth_token, sandbox_url=None):
        self.sandbox_url = sandbox_url or "https://sandbox.yandex-team.ru:443/api/v1.0/"
        self.oauth_header = {"Authorization": "OAuth %s" % oauth_token}

        retries = Retry(backoff_factor=1.0)
        self.session = requests.Session()
        self.session.mount("http://", requests.adapters.HTTPAdapter(max_retries=retries))

    def start_task_async(self, task_type, params, custom_params):
        # create task
        created = requests.post(self.sandbox_url + "task", headers=self.oauth_header, json={"type": task_type})
        if not created.status_code == 201:
            raise Exception("Can't create task", created.text)

        task_id = created.json()["id"]

        task_params = dict(params)
        task_params["custom_fields"] = self.to_custom_fields_format(custom_params)

        # update task params
        info_set = requests.put(self.sandbox_url + "task/" + str(task_id), headers=self.oauth_header, json=task_params)
        if not info_set.status_code == 204:
            raise Exception("Can't set task info", info_set.text)

        # execute tasks
        started = requests.put(self.sandbox_url + "batch/tasks/start", headers=self.oauth_header, json={"id": task_id})

        if not started.status_code == 200 or not all(st["status"] == "SUCCESS" for st in started.json()):
            raise Exception("Can't start task", started.text)

        logging.info("Task %s(%d) is started" % (task_type, task_id))
        return task_id

    def wait_for_task(self, task_id, timeout_in_hours, poll_interval=10):

        start_time = datetime.datetime.now()

        # try to receive terminal status in X hours
        while datetime.datetime.now() < start_time + datetime.timedelta(hours=timeout_in_hours):
            logging.info("Checking task %d status" % task_id)

            task_state = self.fetch_task_state_with_retries(task_id)

            if "status" not in task_state:
                raise Exception("Unexpected response from service:\n%s" % task_state)

            task_status = task_state["status"]

            logging.info("Task %d is %s" % (task_id, task_status))

            if task_status in fail_statuses:
                raise Exception("Sandbox task is %s: %s" % (task_status, task_state))

            elif task_status in success_statuses:
                break

            time.sleep(poll_interval)

        else:  # no terminal status for timeout_in_hours hours:
            raise Exception("Task timed out after %d hours of execution" % timeout_in_hours)

    def fetch_task_state_with_retries(self, task_id, retries_count=60, retry_interval=60):
        task_response = self.session.get(self.sandbox_url + "task/" + str(task_id))

        retries_left = retries_count

        while retries_left > 0:
            try:
                return task_response.json()
            except ValueError:
                response_str = "%s: %s" % (task_response.status_code, task_response.text)
                logging.warning("Can't parse sandbox response " + response_str)
                retries_left -= 1

            time.sleep(retry_interval)

        raise Exception("Couldn't fetch task state in %d retires" % retries_count)

    def to_custom_fields_format(self, custom_params):
        return [{"name": k, "value": v} for k, v in custom_params.items()]
