from __future__ import print_function
import logging

import luigi

import graph_stat
from graph_stat import NAMES, ALL_SOURCE, FINGERPRINT_SOURCE, METRICA_SOURCE, MMETRICA_SOURCE, REQANS_SOURCE
from crypta.graph.v1.python.lib import graphite_sender
from crypta.graph.v1.python.lib.luigi.yt_luigi import PostGraphTask
from crypta.graph.v1.python.matching.human_matching.stats import graph_stat_vertices
from crypta.graph.v1.python.rtcconf import config
from crypta.graph.v1.python.utils import yt_clients

logger = logging.getLogger("graph_statface")

GRAPH_STATFACE_MOBILE_REPORT = "Crypta/%sMatchingMobile"
GRAPH_STATFACE_DESKTOP_REPORT = "Crypta/%sMatchingDesktop"
GRAPH_STATFACE_DEVICE_MOBILE_REPORT = "Crypta/%sMatchingDeviceMobile"
GRAPH_STATFACE_DEVICE_DESKTOP_REPORT = "Crypta/%sMatchingDeviceDesktop"

GRAPH_STATFACE_MOBILE_GEO_REPORT = "Crypta/GeoMobile"
GRAPH_STATFACE_DESKTOP_GEO_REPORT = "Crypta/GeoDesktop"


def read_counts(table, source_columns):
    print("Reading counts from " + table)
    counts_per_source = {}
    yt_client = yt_clients.get_yt_client()
    for rec in yt_client.read_table(table, raw=False):
        count = rec["count"]
        hits = rec["hits"]
        source = "_".join(rec[source_column] for source_column in source_columns if rec[source_column] is not None)
        counts_per_source[source] = count, hits
    return counts_per_source


def to_graphite_old(total_f, sources_f, tables_info, sources, dt):
    for suffix, table, total_table in tables_info:
        total_counts = read_counts(total_f + total_table + "_flatten_count", ["source_types"])
        source_counts = read_counts(sources_f + table + "_flatten_count", ["source_types"])

        for s in sources:
            if s in total_counts and s in source_counts:

                source_count, source_hits = source_counts[s]
                total_count, total_hits = total_counts[s]

                uniqs_metric = NAMES[s] + suffix + "_yuids"
                graphite_sender.to_graphite_sender_batch(
                    uniqs_metric, [("m", str(total_count)), ("md", str(source_count))], dt
                )

                hits_metric = NAMES[s] + suffix + "_traff"
                graphite_sender.to_graphite_sender_batch(
                    hits_metric, [("m", str(total_hits)), ("md", str(source_hits))], dt
                )

                share_metric = NAMES[s] + suffix + "_share"
                uniqs_share = float(source_count) / total_count
                hits_share = float(source_hits) / total_hits

                graphite_sender.to_graphite_sender_batch(
                    share_metric, [("yuids", "%.3f" % uniqs_share), ("traff", "%.3f" % hits_share)], dt
                )


def to_graphite(workdir, metric_name, table_infos, date):
    metric_name = "graph_stat_" + metric_name
    metrics = []
    for prefixes, table, source_columns, flatten in table_infos:
        sum_dir = workdir + "sum_by_" + "_".join(source_columns) + "/"
        prefix_str = ".".join(prefixes)

        if flatten:
            non_unique_counts = read_counts(sum_dir + table + "_flatten_count", source_columns)
            for source, counts in non_unique_counts.iteritems():
                count, hits = counts
                source_name = NAMES[source] if source in NAMES else source

                metrics.append((metric_name, prefix_str + ".non_unique.{}.count".format(source_name), count))
                metrics.append((metric_name, prefix_str + ".non_unique.{}.hits".format(source_name), hits))

            total_counts_recs = read_counts(sum_dir + table + "_count_total", ["total"]).items()
            if total_counts_recs:
                _, total_counts_rec = total_counts_recs[0]
                count, hits = total_counts_rec
                metrics.append((metric_name, prefix_str + ".total.count", count))
                metrics.append((metric_name, prefix_str + ".total.hits", hits))

        else:
            pass

    graphite_sender.to_graphite_sender(metrics, date)


def send_daily_report(dt, report_name, metrics):
    from statface_client import StatfaceClient

    for m in metrics:
        m["fielddate"] = dt  # special name to report date from statface doc

    if config.CRYPTA_ENV == "production" and config.CRYPTA_STATFACE_ENABLED == "yes":
        client = StatfaceClient(
            username=config.STATFACE_USERNAME, oauth_token=config.STATFACE_OAUTH, host="upload.stat.yandex-team.ru"
        )

        report = client.get_report(report_name)
        report.upload_data("daily", metrics)
        logger.info("Uploading to %s:\n%s" % (report_name, metrics))
    else:
        logger.info("Just printing to console %s:\n%s" % (report_name, metrics))


def to_statface(dt, report, part_table, all_table, part_code, all_code, share_code, sources):
    part_counts = read_counts(part_table + "_flatten_count", ["source_types"])
    total_counts = read_counts(all_table + "_flatten_count", ["source_types"])

    metrics = []
    for s in sources:
        if (s in part_counts) and (s in total_counts):
            part_count, part_traf = part_counts[s]
            all_count, all_traf = total_counts[s]

            metric = {
                "service": NAMES[s],
                "metric_type": "uniqs",
                all_code: all_count,
                part_code: part_count,
                share_code: float(part_count) / all_count,
            }
            metrics.append(metric)

            metric = {
                "service": NAMES[s],
                "metric_type": "hits",
                all_code: all_traf,
                part_code: part_traf,
                share_code: float(part_traf) / all_traf,
            }
            metrics.append(metric)
        else:
            logger.warn('No "%s" source in counts' % s)

    send_daily_report(dt, report, metrics)


def to_statface_geo(dt, report, part_table, all_table, part_code, all_code, share_code, sources, geo_ranges):
    metrics = []
    for geo_range in geo_ranges:
        part_counts = read_counts(part_table + "_" + geo_range + "_flatten_count", ["source_types"])
        total_counts = read_counts(all_table + "_flatten_count", ["source_types"])

        if not part_counts or not total_counts:
            continue

        if geo_range == "150":
            report_geo_range = "0150"
        elif geo_range == "500":
            report_geo_range = "0500"
        else:
            report_geo_range = geo_range

        for s in sources:
            part_count, part_traf = part_counts[s]
            all_count, all_traf = total_counts[s]

            metric = {
                "service": NAMES[s],
                "metric_type": "uniqs",
                "geo_range": report_geo_range,
                all_code: all_count,
                part_code: part_count,
                share_code: float(part_count) / all_count,
            }
            metrics.append(metric)

            metric = {
                "service": NAMES[s],
                "metric_type": "hits",
                "geo_range": report_geo_range,
                all_code: all_traf,
                part_code: part_traf,
                share_code: float(part_traf) / all_traf,
            }
            metrics.append(metric)

    send_daily_report(dt, report, metrics)


def report_matching_coverage_statface(date, vertices_prefix, all_folder, crossdev_folder):

    mobile_report = GRAPH_STATFACE_MOBILE_REPORT % vertices_prefix
    desktop_report = GRAPH_STATFACE_DESKTOP_REPORT % vertices_prefix
    device_to_mobile_report = GRAPH_STATFACE_DEVICE_MOBILE_REPORT % vertices_prefix
    device_to_desktop_report = GRAPH_STATFACE_DEVICE_DESKTOP_REPORT % vertices_prefix

    to_statface(
        date,
        mobile_report,
        part_table=crossdev_folder + "mob_yuids_with_desk",
        all_table=all_folder + "mob_yuids_rus",
        part_code="mobile_desktop",
        all_code="mobile",
        share_code="mobile_desktop_percent",
        sources=[FINGERPRINT_SOURCE, REQANS_SOURCE, METRICA_SOURCE, ALL_SOURCE],
    )

    to_statface(
        date,
        desktop_report,
        part_table=crossdev_folder + "desk_yuids_with_mob",
        all_table=all_folder + "desk_yuids_rus",
        part_code="desktop_mobile",
        all_code="desktop",
        share_code="desktop_mobile_percent",
        sources=[FINGERPRINT_SOURCE, REQANS_SOURCE, METRICA_SOURCE, ALL_SOURCE],
    )

    to_statface(
        date,
        device_to_mobile_report,
        part_table=crossdev_folder + "devices_with_mob",
        all_table=all_folder + "devices",
        part_code="devivce_mobile",
        all_code="device",
        share_code="devivce_mobile_percent",
        sources=[MMETRICA_SOURCE],
    )

    to_statface(
        date,
        device_to_desktop_report,
        part_table=crossdev_folder + "devices_with_desk",
        all_table=all_folder + "devices",
        part_code="devivce_desktop",
        all_code="device",
        share_code="devivce_desktop_percent",
        sources=[MMETRICA_SOURCE],
    )


def report_matching_coverage_graphite(date, vertices_prefix, all_folder, crossdev_folder):
    if vertices_prefix:
        # only for compatibility, probably may be removed
        to_graphite_old(
            all_folder + "sum_by_source_types/",
            crossdev_folder + "sum_by_source_types/",
            [("", "mob_yuids_with_desk", "mob_yuids_rus"), ("_desk", "desk_yuids_with_mob", "desk_yuids_rus")],
            [REQANS_SOURCE, METRICA_SOURCE, FINGERPRINT_SOURCE, ALL_SOURCE],
            date,
        )
        to_graphite_old(
            all_folder + "sum_by_source_types/",
            crossdev_folder + "sum_by_source_types/",
            [("_mob", "devices_with_mob", "devices"), ("_desk", "devices_with_desk", "devices")],
            [MMETRICA_SOURCE],
            date,
        )

    metric_name = "cross_device." + vertices_prefix if vertices_prefix else "cross_device"
    to_graphite(
        crossdev_folder,
        metric_name,
        [
            (["mob_yuids_with_desk"], "mob_yuids_with_desk", ["source_types"], True),
            (["desk_yuids_with_mob"], "desk_yuids_with_mob", ["source_types"], True),
            (["devices_with_desk"], "devices_with_desk", ["source_types"], True),
            (["devices_with_mob"], "devices_with_mob", ["source_types"], True),
            (["mob_yuids_with_desk"], "mob_yuids_with_desk", ["source_types"], False),
            (["desk_yuids_with_mob"], "desk_yuids_with_mob", ["source_types"], False),
            (["devices_with_desk"], "devices_with_desk", ["source_types"], False),
            (["devices_with_mob"], "devices_with_mob", ["source_types"], False),
        ],
        date,
    )


def report_geo_statface(date, all_folder, geo_folder):
    to_statface_geo(
        date,
        GRAPH_STATFACE_DESKTOP_GEO_REPORT,
        part_table=geo_folder + "desk_yuids_geo",
        all_table=all_folder + "desk_yuids_rus",
        part_code="desktop_geo",
        all_code="desktop",
        share_code="desktop_geo_percent",
        sources=[ALL_SOURCE],
        geo_ranges=["150", "500", "1000", "all"],
    )

    to_statface_geo(
        date,
        GRAPH_STATFACE_MOBILE_GEO_REPORT,
        part_table=geo_folder + "mob_yuids_geo",
        all_table=all_folder + "mob_yuids_rus",
        part_code="mobile_geo",
        all_code="mobile",
        share_code="mobile_geo_percent",
        sources=[ALL_SOURCE],
        geo_ranges=["150", "500", "1000", "all"],
    )


class MatchingCoverageReportTask(PostGraphTask):
    """
    Reports in-device and cross-device coverage to graphite and statface
    """

    vertices_config = luigi.Parameter()
    legacy_report = luigi.Parameter(
        default=False,
        description="if true, will be stored with no vertices type id in metric name " + "to keep compatibility",
    )
    tags = ["v1"]

    def __init__(self, date, name, vertices_config, legacy_report=False):
        super(MatchingCoverageReportTask, self).__init__(
            date=date, name=name, vertices_config=vertices_config, legacy_report=legacy_report
        )

    def requires(self):
        return [
            graph_stat_vertices.VerticesMatchingCoverageStats(vertices_config=self.vertices_config),
            graph_stat.PrepareTodayTotalUsageStats(self.date),
        ]

    def run_post_graph(self):
        all_folder = config.YT_OUTPUT_FOLDER + self.date + "/stat_new/all/"
        crossdev_folder = self.vertices_config.get_vertices_folder() + "stat/matching/cross_device/"

        if self.legacy_report:
            vertices_prefix_statface = ""
            vertices_prefix_graphite = ""
        else:
            vertices_prefix_statface = self.vertices_config.vertices_type + "/"
            vertices_prefix_graphite = self.vertices_config.vertices_type

        report_matching_coverage_statface(
            self.date,
            vertices_prefix_statface,
            all_folder + "sum_by_source_types/",
            crossdev_folder + "sum_by_source_types/",
        )

        report_matching_coverage_graphite(self.date, vertices_prefix_graphite, all_folder, crossdev_folder)


def copy_report_config(from_r, to_r):
    from statface_client import StatfaceClient

    client = StatfaceClient(
        username=config.STATFACE_USERNAME, oauth_token=config.STATFACE_OAUTH, host="stat.yandex-team.ru"
    )

    old = client.get_old_report(from_r)
    client.get_new_report(to_r).upload_config(old.config)
