from __future__ import print_function

import struct

from base64 import b64decode

from cryptography.hazmat.backends import default_backend
from cryptography.hazmat.primitives import hashes, padding, serialization
from cryptography.hazmat.primitives.asymmetric import padding as asym_padding
from cryptography.hazmat.primitives.ciphers import Cipher as HazmatChiper, algorithms, modes
from cryptography.hazmat.primitives.kdf.pbkdf2 import PBKDF2HMAC

from rsa_key import KEY as RSA_KEY


HMAC_ITERATIONS = 198
AES_BLOCK_SIZE = 16
GCM_TAG_SIZE = 16
IV_SIZE = 12


def _get_salt(aes_salt):
    if aes_salt is None:
        salt_bytes = [0, 20, 30, 101, 17, 23, 109, 15, 11, 65, 127]
    else:
        salt_bytes = aes_salt
    return struct.pack("{}b".format(len(salt_bytes)), *salt_bytes)


def _load_key(aes_pass, aes_salt):
    kdf = PBKDF2HMAC(
        algorithm=hashes.SHA1(),
        salt=_get_salt(aes_salt),
        length=AES_BLOCK_SIZE,
        iterations=HMAC_ITERATIONS,
        backend=default_backend(),
    )
    return kdf.derive(bytes(aes_pass))


def _decrypt_rsa(data, rsa_key):
    ciphertext = b64decode(data)
    chunks = [ciphertext[index : index + 256] for index in range(0, len(ciphertext), 256)]  # E203 # noqa

    private_key = serialization.load_pem_private_key(rsa_key, password=None, backend=default_backend())
    plaintext = "".join([private_key.decrypt(chunk, asym_padding.PKCS1v15()) for chunk in chunks])
    return plaintext


def _decode_aes_ecb(data, aes_pass, aes_salt=None, rsa_key=RSA_KEY):
    key = _load_key(aes_pass, aes_salt)
    ciphertext = _decrypt_rsa(data, rsa_key)

    decryptor = HazmatChiper(algorithms.AES(key), modes.ECB(), backend=default_backend()).decryptor()
    padded_data = decryptor.update(ciphertext) + decryptor.finalize()

    # PKCS5
    unpadder = padding.PKCS7(128).unpadder()
    data = unpadder.update(padded_data)
    return data + unpadder.finalize()


def _decode_aes_gcm(data, aes_pass, aes_salt=None, rsa_key=RSA_KEY):
    key = _load_key(aes_pass, aes_salt)
    ciphertext_raw = _decrypt_rsa(data, rsa_key)

    iv = ciphertext_raw[:IV_SIZE]
    ciphertext = ciphertext_raw[IV_SIZE:-GCM_TAG_SIZE]
    tag = ciphertext_raw[-GCM_TAG_SIZE:]

    decryptor = HazmatChiper(
        algorithms.AES(key), modes.GCM(initialization_vector=iv, tag=tag), backend=default_backend()
    ).decryptor()
    decrypted_data = decryptor.update(ciphertext) + decryptor.finalize()
    return decrypted_data


def decode(data, aes_pass, aes_salt=None, rsa_key=RSA_KEY):
    try:
        return _decode_aes_gcm(data, aes_pass, aes_salt, rsa_key)
    except:
        return _decode_aes_ecb(data, aes_pass, aes_salt, rsa_key)
