from functools import partial

import sys
import luigi

from crypta.graph.v1.python.matching.device_matching.app_metrica import app_metrica_dict_schemas
from crypta.graph.v1.python.matching.device_matching.app_metrica import app_metrica_month
from crypta.graph.v1.python.lib.luigi import yt_luigi
from crypta.graph.v1.python.rtcconf import config
from crypta.graph.v1.python.utils import mr_utils as mr
from crypta.graph.v1.python.utils import utils
from crypta.graph.v1.python.v2 import ids

from crypta.graph.soup.config.python import ID_TYPE


def to_mac_address(rec):
    for mac in rec["macs"]:
        yield {"devid": rec[ids.CRYPTA_DEVICE_ID], "mac": mac, "ts": rec["ts"]}


def to_mmetric_hash(rec):
    for mmetric_devid, mmetric_devid_hash in rec["mmetric_device_ids_to_hash"].iteritems():
        yield {"devid": rec[ids.CRYPTA_DEVICE_ID], "mmetric_devid": mmetric_devid, "devidhash": mmetric_devid_hash}


def to_yamr_format(rec_key, rec):
    yamr_value = "\t".join([k + "=" + (str(v) if v else "") for k, v in rec.iteritems()])
    return {"key": rec_key, "subkey": "", "value": yamr_value}


def to_dev_info_format(rec, only_yamr_rec=False):
    out_rec = {
        "devid": rec[ids.CRYPTA_DEVICE_ID],
        "idfa": rec[ID_TYPE.IDFA.Name],
        "android_id": rec[ids.ANDROID_ID],
        "google_adv_id": rec[ids.GOOGLE_ADV_ID],
        "mmetric_devids": ",".join(rec["mmetric_device_ids"] or []),
        "apps": ",".join(set((rec["metrika_apps"] or []) + (rec["ssp_apps"] or []))),
        "ssp_apps": ",".join(rec["ssp_apps"] or []),
        "device_type": rec["device_type"],
        "manufacturer": rec["manufacturer"],
        "model": rec["model"],
        "os": rec["os"],
        "os_version": rec["os_version"],
        "ua_profile": rec["ua_profile"],
        "locale": rec["locale"],
        "connection_hist": rec["connection_hist"],
        "ts": rec["ts"],
        "dates": rec["dates"],
        "subkey": str(sys.maxint - abs(rec["ts"])),
    }

    yamr_rec = to_yamr_format(rec[ids.CRYPTA_DEVICE_ID], out_rec)

    if only_yamr_rec:
        yield yamr_rec

    else:
        yield out_rec

        yamr_rec["@table_index"] = 1
        yield yamr_rec


def to_uuid_info_format(rec, only_yamr_rec=False):
    out_rec = {
        "uuid": rec[ID_TYPE.UUID.Name],
        "api_keys": rec["api_keys"],
        "app_id": rec["app_id"],
        "app_version": rec["app_version"],
        "devid": rec[ids.CRYPTA_DEVICE_ID],
        "devidhash": rec[ids.MMETRIC_DEVICE_ID + ids.HASH_POSTFIX],
        "idfa": rec[ID_TYPE.IDFA.Name],
        "android_id": rec[ids.ANDROID_ID],
        "google_adv_id": rec[ids.GOOGLE_ADV_ID],
        "mmetric_devid": rec[ids.MMETRIC_DEVICE_ID],
        "ua_profile": rec["ua_profile"],
        "ts": rec["ts"],
        "subkey": str(sys.maxint - abs(rec["ts"])),
    }

    yamr_rec = to_yamr_format(rec[ID_TYPE.UUID.Name], out_rec)

    if only_yamr_rec:
        yield yamr_rec

    else:
        yield out_rec

        yamr_rec["@table_index"] = 1
        yield yamr_rec


def reduce_first_rec(_, recs):
    for rec in recs:
        rec["@table_index"] = 0
        yield rec
        return


class ConvertAppMetricaDictsToOldFormats(yt_luigi.BaseYtTask):
    date = luigi.Parameter()
    tags = ["v1"]

    def input_folders(self):
        return {"ids_storage": config.CRYPTA_IDS_STORAGE}

    def output_folders(self):
        return {"dict": config.GRAPH_YT_DICTS_FOLDER, "mobile": config.GRAPH_YT_OUTPUT_FOLDER + self.date + "/mobile/"}

    def requires(self):
        return app_metrica_month.AppMetricaDictMergeMonthTask(self.date)

    def run(self):
        with self.yt.Transaction() as tr:
            for t in ["devid_hash", "devid_mac", "dev_info_yt", "uuid_dev_info_yt"]:
                mr.create_table_with_schema(
                    self.out_f("dict") + t,
                    app_metrica_dict_schemas.dict_table_schemas[t],
                    tr,
                    strict=True,
                    recreate_if_exists=True,
                    # sorted_by=app_metrica_dict_schemas.dict_schemas_sort_keys[t]
                )

            utils.wait_all(
                [
                    self.yt.run_map_reduce(
                        to_mmetric_hash,
                        partial(mr.distinct_mr, distinct_keys=["mmetric_devid", "devidhash", "devid"]),
                        self.in_f("ids_storage") + "device_id/app_metrica_month",
                        self.out_f("dict") + "devid_hash",
                        reduce_by=["devidhash", "mmetric_devid", "devid"],
                        sync=False,
                    ),
                    self.yt.run_map_reduce(
                        to_mac_address,
                        reduce_first_rec,
                        self.in_f("ids_storage") + "device_id/app_metrica_month",
                        self.out_f("mobile") + "devid_mac_month_tmp",
                        sort_by=["mac", "devid", "ts"],
                        reduce_by=["mac", "devid"],
                        sync=False,
                    ),
                    self.yt.run_map(
                        to_dev_info_format,
                        self.in_f("ids_storage") + "device_id/app_metrica_month",
                        [self.out_f("dict") + "dev_info_yt", self.out_f("dict") + "dev_info"],
                        ordered=True,
                        sync=False,
                    ),
                    self.yt.run_map(
                        to_uuid_info_format,
                        self.in_f("ids_storage") + "uuid/app_metrica_month",
                        [self.out_f("dict") + "uuid_dev_info_yt", self.out_f("dict") + "uuid_dev_info"],
                        ordered=True,
                        sync=False,
                    ),
                ]
            )

            # devid-mac should be 1-1 relationship to use it in matching
            from crypta.graph.v1.python.matching.device_matching.perfect.enrich import devid_mac_vmetro

            self.yt.run_map_reduce(
                None,
                devid_mac_vmetro.filter_multi_devid_macs,
                self.out_f("mobile") + "devid_mac_month_tmp",
                [self.out_f("dict") + "devid_mac", self.out_f("mobile") + "mac_multi_devid"],
                reduce_by="mac",
            )

            utils.wait_all(
                [
                    self.yt.run_sort(self.out_f("dict") + "dev_info", sort_by="key", sync=False),
                    self.yt.run_sort(self.out_f("dict") + "dev_info_yt", sort_by="devid", sync=False),
                    self.yt.run_sort(self.out_f("dict") + "uuid_dev_info", sort_by="key", sync=False),
                    self.yt.run_sort(self.out_f("dict") + "uuid_dev_info_yt", sort_by="uuid", sync=False),
                    self.yt.run_sort(self.out_f("dict") + "devid_hash", sort_by="devid", sync=False),
                    self.yt.run_sort(
                        self.out_f("dict") + "devid_hash",
                        self.out_f("dict") + "mmetric_devid",
                        sort_by="mmetric_devid",
                        sync=False,
                    ),
                    self.yt.run_sort(self.out_f("dict") + "devid_mac", sort_by="mac", sync=False),
                ]
            )

            mr.drop(self.out_f("mobile") + "devid_mac_month_tmp")

            mr.set_generate_date(self.out_f("dict") + "dev_info", self.date)
            mr.set_generate_date(self.out_f("dict") + "dev_info_yt", self.date)
            mr.set_generate_date(self.out_f("dict") + "uuid_dev_info", self.date)
            mr.set_generate_date(self.out_f("dict") + "uuid_dev_info_yt", self.date)
            mr.set_generate_date(self.out_f("dict") + "devid_hash", self.date)
            mr.set_generate_date(self.out_f("dict") + "mmetric_devid", self.date)
            mr.set_generate_date(self.out_f("dict") + "devid_mac", self.date)

    def output(self):
        return [
            yt_luigi.YtDateTarget(t, self.date)
            for t in [
                self.out_f("dict") + "dev_info",
                self.out_f("dict") + "dev_info_yt",
                self.out_f("dict") + "uuid_dev_info",
                self.out_f("dict") + "devid_hash",
                self.out_f("dict") + "uuid_dev_info_yt",
                self.out_f("dict") + "mmetric_devid",
                self.out_f("dict") + "devid_mac",
            ]
        ]
