import luigi

from crypta.graph.v1.python.data_imports.import_logs.graph_eal import ImportEalDayTask
from crypta.graph.v1.python.lib.luigi import yt_luigi
from crypta.graph.v1.python.rtcconf import config
from crypta.graph.v1.python.utils import mr_utils as mr, utils
from crypta.graph.v1.python.utils import yt_clients


def reduce_ui_yuid_all(key, recs):
    yuid_date = {}
    for rec in recs:
        yuid = rec["yuid"]
        # today records
        if rec["@table_index"] == 0:
            yuid_date[yuid] = rec["id_date"]
        else:
            # remove new yuids if old present, to make sure we yield earliest ui date
            yuid_date.pop(yuid, "")
            rec["@table_index"] = 0
            yield rec

    for yuid, id_date in yuid_date.iteritems():
        yield {"yuid": yuid, "id_value": key["id_value"], "id_date": id_date}


def reduce_add_browser_logins(_, recs):
    yuid_logins = []
    yuid_browser = ""
    for rec in recs:
        if rec["@table_index"] == 0:
            login_fp_dates = rec.get("login_fp_dates") or {}
            yuid_logins = list(login_fp_dates.keys())
            yuid_browser = rec.get("browser") or ""
        elif rec["@table_index"] == 1:
            rec["yuid_logins"] = yuid_logins
            rec["yuid_browser"] = yuid_browser
            rec["@table_index"] = 0
            yield rec


def run_ui_yuid_all(eal_dt_table, workdir, dicts_in_folder, dicts_out_folder, dt):
    tables = [eal_dt_table]
    yt_client = yt_clients.get_yt_client()
    if yt_client.exists(dicts_in_folder + "ui_yuid_all"):
        tables.append(dicts_in_folder + "ui_yuid_all")

    tmp_table = workdir + "ui_yuid_all_tmp"

    schema = [
        {"name": "id_value", "sort_order": "ascending", "required": False, "type": "string"},
        {"name": "id_count", "required": False, "type": "int64"},
        {"name": "id_date", "required": False, "type": "string"},
        {"name": "id_type", "required": False, "type": "string"},
        {"name": "source_type", "required": False, "type": "string"},
        {"name": "yuid", "required": False, "type": "string"},
        {"name": "yuid_browser", "required": False, "type": "string"},
        {"name": "yuid_logins", "required": False, "type": "any"},
    ]
    yt_client.create("table", tmp_table, recursive=True, force=True, attributes={"schema": schema})
    yt_client.run_sort(tables, tmp_table, sort_by="id_value", spec={"schema_inference_mode": "from_output"})
    yt_client.run_reduce(reduce_ui_yuid_all, tmp_table, workdir + "ui_yuid_all", reduce_by="id_value")
    yt_client.remove(tmp_table)

    yt_client.run_sort(
        workdir + "ui_yuid_all",
        sort_by="yuid",
        spec={"combine_chunks": True},
    )
    yt_client.run_reduce(
        reduce_add_browser_logins,
        [
            yt_client.TablePath(dicts_in_folder + "yuid_with_all", columns=["yuid", "login_fp_dates", "browser"]),
            workdir + "ui_yuid_all",
        ],
        workdir + "ui_yuid_all",
        reduce_by="yuid",
    )

    yt_client.run_sort(
        workdir + "ui_yuid_all",
        dicts_out_folder + "ui_yuid_all",
        sort_by="id_value",
        spec={"combine_chunks": True},
    )

    mr.merge(workdir + "ui_yuid_all")
    mr.set_generate_date(dicts_out_folder + "ui_yuid_all", dt)


class UiYuidAllTask(yt_luigi.BaseYtTask):
    date = luigi.Parameter()
    tags = ["v1"]

    def input_folders(self):
        return {"yuid_raw": config.YT_OUTPUT_FOLDER + self.date + "/yuid_raw/", "dict": config.GRAPH_YT_DICTS_FOLDER}

    def output_folders(self):
        return {"yuid_raw": config.YT_OUTPUT_FOLDER + self.date + "/yuid_raw/", "dict": config.GRAPH_YT_DICTS_FOLDER}

    def requires(self):
        import crypta.graph.v1.python.matching.yuid_matching.graph_dict as graph_dict

        return [
            ImportEalDayTask(date=d, run_date=self.date)
            for d in utils.get_dates_before(self.date, int(config.STORE_DAYS))
        ] + [graph_dict.YuidAllIdBySourceDictsTask(self.date)]

    def before_run(self):
        mr.mkdir(self.out_f("yuid_raw"))

    def run(self):
        run_ui_yuid_all(
            self.in_f("yuid_raw") + "yuid_with_" + config.ID_SOURCE_TYPE_EAL,
            self.out_f("yuid_raw"),
            self.in_f("dict"),
            self.out_f("dict"),
            self.date,
        )

    def output(self):
        return [yt_luigi.YtDateTarget(self.out_f("dict") + "ui_yuid_all", self.date)]
