from collections import defaultdict

import luigi

from crypta.graph.v1.python.data_imports.import_logs import (
    device_yuids_oauth,
    device_yuids_redir,
    device_yuids_sdk,
    device_yuids_tracking,
    graph_access,
    graph_barnavig,
    graph_postclicks,
    graph_watch_log,
)
from crypta.graph.v1.python.lib.luigi import yt_luigi
from crypta.graph.v1.python.matching.device_matching.perfect.config import merge_configs_daily
from crypta.graph.v1.python.rtcconf import config
from crypta.graph.v1.python.utils import mr_utils as mr, utils
from crypta.graph.v1.python.utils import yt_clients


def merge_day_events_month(key, recs):
    dates_hits = defaultdict(int)
    last_ts = None

    for r in recs:
        dates_hits[r["dt"]] += r["hits_count"]

        rec_ts = r.get("last_ts")  # comes in reverse order, thus we store first
        if not last_ts and rec_ts:
            last_ts = rec_ts

    out_rec = dict(key)
    out_rec["last_ts"] = last_ts
    out_rec["dates"] = utils.default_to_regular(dates_hits)
    yield out_rec


def run_merge_all_dates(device_base_f, workdir, date):
    for m in merge_configs_daily:
        m.day_tables = mr.get_existing_date_tables(
            device_base_f, "perfect/devid_raw_day/" + m.table_name(), int(config.STORE_DAYS), before_date=date
        )
        if not m.day_tables:
            raise Exception("Can't find any day table for " + m.table_name())

    # produce separate month table for every pair type
    yt_client = yt_clients.get_yt_client()
    utils.wait_all(
        [
            yt_client.run_map_reduce(
                None,
                merge_day_events_month,
                m.day_tables,
                workdir + "devid_raw_month/" + m.table_name(),
                reduce_by=[m.id1, m.id2, "source_type"],
                sort_by=[m.id1, m.id2, "source_type", "last_ts"],
                sync=False,
            )
            for m in merge_configs_daily
        ]
    )


class DeviceYuidsMergeMonthTask(yt_luigi.BaseYtTask):
    date = luigi.Parameter()
    tags = ["v1"]

    def input_folders(self):
        return {"indevice": config.INDEVICE_YT_FOLDER}

    def output_folders(self):
        return {"perfect": config.INDEVICE_YT_FOLDER + self.date + "/perfect/"}

    def requires(self):
        day_tasks = []
        for dt in utils.get_dates_before(self.date, int(config.STORE_DAYS)):
            day_tasks.append(device_yuids_redir.ImportRedirLogsDayTask(date=dt, run_date=self.date))
            day_tasks.append(device_yuids_oauth.ImportOauthLogsDayTask(date=dt, run_date=self.date))
            day_tasks.append(graph_access.ImportAccessLogsDayTask(date=dt, run_date=self.date))
            day_tasks.append(graph_barnavig.ImportBarNavigDayTask(date=dt, run_date=self.date))
            day_tasks.append(graph_watch_log.ImportWatchLogDayTask(date=dt, run_date=self.date))
            day_tasks.append(device_yuids_tracking.ImportMobileTrackingLogsDayTask(date=dt, run_date=self.date))
            day_tasks.append(device_yuids_sdk.ImportSdkLogsDayTask(date=dt, run_date=self.date))
            day_tasks.append(graph_postclicks.ImportPostclicksDailyTask(date=dt, run_date=self.date))

        return day_tasks

    def before_run(self):
        mr.mkdir(self.out_f("perfect") + "devid_raw_month")

    def run(self):
        run_merge_all_dates(self.in_f("indevice"), self.out_f("perfect"), self.date)

    def output(self):
        return [
            yt_luigi.YtTarget(self.out_f("perfect") + "devid_raw_month/" + m.table_name()) for m in merge_configs_daily
        ]
