import os


class Lock(object):

    """ Lock file decorator """

    def __init__(self, lockfile=None, logger=None):
        self.lockfile = lockfile or "/var/run/lock/.lock"
        self.logger = logger

    def __call__(self, function):
        def wrapped(*args, **kwargs):
            if self.is_locked():
                self.log("Task locked. Exit.")
                return
            try:
                self._lock()
                return function(*args, **kwargs)
            except:
                raise
            finally:
                self._unlock()

        return wrapped

    def is_locked(self):
        return os.path.exists(self.lockfile)

    def _lock(self):
        self.log("Lock task.")
        self._touch(self.lockfile)

    def _unlock(self):
        if self.is_locked():
            os.remove(self.lockfile)
        self.log("Unlock task.")

    def _touch(self, file_path):
        basedir = os.path.dirname(file_path)
        if not os.path.exists(basedir):
            os.makedirs(basedir)
        with open(file_path, "a"):
            os.utime(file_path, None)

    def log(self, message):
        if self.logger is not None:
            self.logger.info(message)
