import logging
import logging.handlers
from logging.handlers import SysLogHandler

import luigi  # noqa

from crypta.graph.v1.python.rtcconf import config


THE_LOG_FORMAT = "%(levelname)s:%(filename)s:%(process)d:%(thread)d %(asctime)s %(message)s"


def default_handler():
    handler = getattr(default_handler, "default_handler", None)
    if not handler:
        handler = SysLogHandler(address="/dev/log", facility="local3")
        handler.setLevel(logging.INFO)
        handler.setFormatter(logging.Formatter(THE_LOG_FORMAT))
        setattr(default_handler, "default_handler", handler)

    return handler


def setup_root_logger():
    root_logger = logging.getLogger()
    root_logger.setLevel(logging.INFO)
    root_logger.addHandler(default_handler())


def _setup_handlers(logger, prefix):
    import logging.handlers

    info_handler = logging.handlers.TimedRotatingFileHandler(
        config.LOG_FOLDER + prefix + "/info.log", when="MIDNIGHT", backupCount=10
    )
    info_handler.setLevel(logging.INFO)
    info_handler.setFormatter(logging.Formatter(THE_LOG_FORMAT))
    logger.addHandler(info_handler)

    error_handler = logging.handlers.TimedRotatingFileHandler(
        config.LOG_FOLDER + prefix + "/error.log", when="MIDNIGHT", backupCount=10
    )
    error_handler.setLevel(logging.ERROR)
    error_handler.setFormatter(logging.Formatter(THE_LOG_FORMAT))
    logger.addHandler(error_handler)


def setup_luigi_logging():
    """
    * redirect luigi info+ messages to luigi/info.log
    * redirect luigi error+ messages to luigi/error.log
    """

    class LuigiFilter(logging.Filter):
        """
        mute meaningless logging ("running" messages every second)
        """

        def filter(self, record):
            if (
                record.msg == "Done"
                or record.msg == "There are no more tasks to run at this time"
                or "is currently run by worker" in record.msg
            ):
                return 0
            return 1

    logger = logging.getLogger("luigi-interface")
    logger.setLevel(logging.INFO)
    logger.addFilter(LuigiFilter())
    _setup_handlers(logger, "luigi")

    # luigi.interface.setup_interface_logging.has_run = True


def setup_yt_logging():
    logger = logging.getLogger("Yt")
    _setup_handlers(logger, "yt")
    logger.removeHandler(logger.handlers[0])  # remove printing to stderr
    logger.addHandler(default_handler())  # yt disables message propagation, so we need to set handler explicitly

    # TODO: remove stream handler from mapreducelib logger
    mapreducelib_logger = logging.getLogger("mapreducelib.default")
    _setup_handlers(mapreducelib_logger, "yt")
    mapreducelib_logger.addHandler(default_handler())
