import luigi

from os.path import join as ypj
from datetime import datetime, timedelta

from library.python import resource

from crypta.graph.v1.python.rtcconf import config
from crypta.graph.soup.config.python import ID_TYPE as ids, LOG_SOURCE as log_sources  # noqa
from crypta.graph.v1.python.lib.luigi import yt_luigi
from crypta.graph.v1.python.utils import mr_utils as mr, utils
from crypta.graph.v1.python.utils.yql_utils import run_yql
from crypta.graph.v1.python.data_imports.import_logs.app_metrica_day import ImportAppMetrikaDayTask
from crypta.graph.v1.python.data_imports.import_logs.graph_watch_log import ImportWatchLogDayTask
from crypta.graph.v1.python.v2.soup import soup_dirs
from crypta.graph.v1.python.data_imports import StreamImportConfig
from crypta.graph.v1.python.utils import yt_clients


DEVICE_IDS = [ids.IDFA, ids.IFV, ids.OAID, ids.GAID, ids.MM_DEVICE_ID]


def idstorage_path(basepath, idtype, kind="eternal"):
    return ypj(basepath, idtype.Name, kind)


def get_throw_before_date(dt):
    if config.CRYPTA_ENV == "testing" or config.CRYPTA_ENV == "development":
        ttl_date = datetime.strptime(dt, "%Y-%m-%d") - timedelta(days=config.SOUP_TESTING_TTL_DAYS)
        return ttl_date.strftime("%Y-%m-%d")
    return None


def merge_cookies(idt, idstorage_table, dt, tx):
    input_tables = []
    remove_input = False

    yt_client = yt_clients.get_yt_client()
    if log_sources.WATCH_LOG.Name in config.STREAMING_LOGS:
        folder = "{}/{}".format(StreamImportConfig().paths.stream.id_storage, idt.Name)
        input_tables = yt_client.list(folder, absolute=True)
        remove_input = True
    else:
        daily_wl = soup_dirs.get_day_log_dir(dt)
        input_tables = [ypj(daily_wl, "idstorage_" + idt.Name)]

    with yt_client.TempTable() as tmp:
        context = dict(
            tx=str(tx.transaction_id),
            date=dt,
            day_tables=input_tables,
            idstorage_table=idstorage_table,
            out=tmp,
            id_type=idt.Name,
            soup_root=soup_dirs.SOUP_DIR.rstrip("/"),
            throw_before_date=get_throw_before_date(dt),
            soup_dates_enabled=config.SOUP_DATES_ENABLED,
        )

        run_yql(query=resource.find("/ids_storage/cookie.sql.j2"), context=context)
        yt_client.copy(tmp, idstorage_table, force=True)

        if remove_input:
            for t in input_tables:
                yt_client.remove(t)


def merge_duids(idt, idstorage_table, dt, tx):
    input_tables = []
    remove_input = False

    yt_client = yt_clients.get_yt_client()
    if log_sources.WATCH_LOG.Name in config.STREAMING_LOGS:
        folder = "{}/{}".format(StreamImportConfig().paths.stream.id_storage, idt.Name)
        input_tables = yt_client.list(folder, absolute=True)
        remove_input = True
    else:
        daily_wl = soup_dirs.get_day_log_dir(dt)
        input_tables = [ypj(daily_wl, "idstorage_" + idt.Name)]

    with yt_client.TempTable() as tmp:
        context = dict(
            min_active_date=datetime.strptime(dt, "%Y-%m-%d") - timedelta(days=180),
            tx=str(tx.transaction_id),
            date=dt,
            day_tables=input_tables,
            idstorage_table=idstorage_table,
            out=tmp,
        )

        run_yql(query=resource.find("/ids_storage/duids.sql.j2"), context=context)
        yt_client.copy(tmp, idstorage_table, force=True)

        if remove_input:
            for t in input_tables:
                yt_client.remove(t)


def merge_devices(day_table, idstorage, dt, tx):
    context = dict(
        tx=str(tx.transaction_id),
        date=dt,
        day_table=day_table,
        idstorage_tables={id_type.Name: idstorage_path(idstorage, id_type) for id_type in DEVICE_IDS},
        soup_root=soup_dirs.SOUP_DIR.rstrip("/"),
        soup_dates_enabled=config.SOUP_DATES_ENABLED,
    )

    run_yql(query=resource.find("/ids_storage/devices.sql.j2"), context=context)


def merge_apps(day_table, idstorage_table, dt, tx):
    yt_client = yt_clients.get_yt_client()
    with yt_client.TempTable() as tmp:
        context = dict(
            tx=str(tx.transaction_id),
            date=dt,
            day_table=day_table,
            idstorage_table=idstorage_table,
            tmp=tmp,
            id_type=ids.UUID.Name,
            soup_root=soup_dirs.SOUP_DIR.rstrip("/"),
            soup_dates_enabled=config.SOUP_DATES_ENABLED,
        )

        run_yql(query=resource.find("/ids_storage/apps.sql.j2"), context=context)

        yt_client.copy(tmp, idstorage_table, force=True)


def create_tables(basepath):
    cookie_schema = {
        "id": "string",
        "id_type": "string",
        "date_begin": "string",
        "date_end": "string",
        "browser_name": "string",
        "browser_version": "string",
        "os_name": "string",
        "os_family": "string",
        "os_version": "string",
        "is_emulator": "boolean",
        "is_browser": "boolean",
        "is_mobile": "boolean",
        "is_tablet": "boolean",
        "is_touch": "boolean",
        "is_robot": "boolean",
        "is_tv": "boolean",
    }

    device_schema = {
        "id": "string",
        "id_type": "string",
        "date_begin": "string",
        "date_end": "string",
        "manufacturer": "string",
        "model": "string",
        "os": "string",
        "os_version": "string",
        "screen_width": "uint16",
        "screen_height": "uint16",
    }

    app_schema = {
        "id": "string",
        "id_type": "string",
        "date_begin": "string",
        "date_end": "string",
        "app_id": "string",
        "app_version": "string",
        "app_platform": "string",
        "api_keys": "any",
    }

    schemas = [(id_type, device_schema) for id_type in DEVICE_IDS] + [
        (ids.YANDEXUID, cookie_schema),
        (ids.ICOOKIE, cookie_schema),
        (ids.UUID, app_schema),
    ]

    yt_client = yt_clients.get_yt_client()
    with yt_client.Transaction() as tx:
        for idt, sch in schemas:
            tbl = idstorage_path(basepath, idt)
            if not yt_client.exists(tbl):
                mr.create_table_with_schema(tbl, sch, tx)


class InitStorageTables(yt_luigi.BaseYtTask):
    def output_folders(self):
        return {"ids_storage": config.CRYPTA_IDS_STORAGE}

    def output(self):
        ids_storage_base = self.out_f("ids_storage")
        return [
            yt_luigi.YtTarget(idstorage_path(ids_storage_base, x), allow_empty=True)
            for x in DEVICE_IDS + [ids.UUID, ids.YANDEXUID, ids.ICOOKIE]
        ]

    def run(self):
        create_tables(self.out_f("ids_storage"))


class UpdateIdStorageYandexuid(yt_luigi.BaseYtTask):
    date = luigi.Parameter()

    def input_folders(self):
        return {"wl": soup_dirs.get_day_log_dir(self.date)}

    def output_folders(self):
        return {"ids_storage": config.CRYPTA_IDS_STORAGE}

    def requires(self):
        return [InitStorageTables(), ImportWatchLogDayTask(self.date, self.date)]

    def output(self):
        ids_storage_f = self.out_f("ids_storage")
        return yt_luigi.YtDateTarget(idstorage_path(ids_storage_f, ids.YANDEXUID), self.date)

    def run(self):
        with self.yt.Transaction() as tx:
            idt = ids.YANDEXUID
            out_path = idstorage_path(self.out_f("ids_storage"), idt)
            merge_cookies(idt, out_path, self.date, tx)
            mr.set_generate_date(out_path, self.date)


class UpdateIdStorageICookie(yt_luigi.BaseYtTask):
    date = luigi.Parameter()

    def input_folders(self):
        return {"wl": soup_dirs.get_day_log_dir(self.date)}

    def output_folders(self):
        return {"ids_storage": config.CRYPTA_IDS_STORAGE}

    def requires(self):
        return [InitStorageTables(), ImportWatchLogDayTask(self.date, self.date)]

    def output(self):
        ids_storage_f = self.out_f("ids_storage")
        return yt_luigi.YtDateTarget(idstorage_path(ids_storage_f, ids.ICOOKIE), self.date)

    def run(self):
        with self.yt.Transaction() as tx:
            idt = ids.ICOOKIE
            out_path = idstorage_path(self.out_f("ids_storage"), idt)
            merge_cookies(idt, out_path, self.date, tx)
            mr.set_generate_date(out_path, self.date)


class UpdateIdStorageDuid(yt_luigi.BaseYtTask):
    date = luigi.Parameter()

    def input_folders(self):
        return {"wl": soup_dirs.get_day_log_dir(self.date)}

    def output_folders(self):
        return {"ids_storage": config.CRYPTA_IDS_STORAGE}

    def requires(self):
        return [InitStorageTables(), ImportWatchLogDayTask(self.date, self.date)]

    def output(self):
        idt = ids.DUID
        out_path = idstorage_path(self.out_f("ids_storage"), idt, kind="index")
        return yt_luigi.YtDateTarget(out_path, self.date)

    def run(self):
        with self.yt.Transaction() as tx:
            idt = ids.DUID
            out_path = idstorage_path(self.out_f("ids_storage"), idt, kind="index")
            merge_duids(idt, out_path, self.date, tx)
            mr.set_generate_date(out_path, self.date)


class UpdateIdStorageApps(yt_luigi.BaseYtTask):
    date = luigi.Parameter()

    def input_folders(self):
        return {"appm": ImportAppMetrikaDayTask(self.date, self.date).output_folders()["mobile"]}

    def output_folders(self):
        return {"ids_storage": config.CRYPTA_IDS_STORAGE}

    def requires(self):
        return [InitStorageTables(), ImportAppMetrikaDayTask(self.date, self.date)]

    def output(self):
        ids_storage_f = self.out_f("ids_storage")
        id_types = [ids.UUID]
        return [yt_luigi.YtDateTarget(idstorage_path(ids_storage_f, x), self.date) for x in id_types]

    def run(self):
        with self.yt.Transaction() as tx:
            out_path = idstorage_path(self.out_f("ids_storage"), ids.UUID)
            merge_apps(ypj(self.in_f("appm"), "uuid_info_yt"), out_path, self.date, tx)
            mr.set_generate_date(out_path, self.date)


class UpdateIdStorageDevices(yt_luigi.BaseYtTask):
    date = luigi.Parameter()

    def input_folders(self):
        return {"appm": ImportAppMetrikaDayTask(self.date, self.date).output_folders()["mobile"]}

    def output_folders(self):
        return {"ids_storage": config.CRYPTA_IDS_STORAGE}

    def requires(self):
        return [InitStorageTables(), ImportAppMetrikaDayTask(self.date, self.date)]

    def output(self):
        ids_storage_f = self.out_f("ids_storage")
        return [yt_luigi.YtDateTarget(idstorage_path(ids_storage_f, x), self.date) for x in DEVICE_IDS]

    def run(self):
        with self.yt.Transaction() as tx:
            merge_devices(ypj(self.in_f("appm"), "dev_info_yt"), self.out_f("ids_storage"), self.date, tx)
            for idt in DEVICE_IDS:
                out_path = idstorage_path(self.out_f("ids_storage"), idt)
                mr.set_generate_date(out_path, self.date)


class UpdateEternalIdStorage(yt_luigi.BaseYtTask):

    date = luigi.Parameter()

    def output_folders(self):
        return {"ids_storage": config.CRYPTA_IDS_STORAGE}

    @property
    def id_types(self):
        return DEVICE_IDS + [ids.UUID, ids.YANDEXUID, ids.ICOOKIE]

    def requires(self):
        return [
            UpdateIdStorageYandexuid(self.date),
            UpdateIdStorageICookie(self.date),
            UpdateIdStorageDuid(self.date),
            UpdateIdStorageDevices(self.date),
            UpdateIdStorageApps(self.date),
        ]

    def run(self):
        template = """
            PRAGMA yt.ExternalTx = "{tx.transaction_id}";

            DEFINE ACTION $fresh($id) AS
                $source = String::ReplaceAll({in_path!r}, "X_ID", $id);
                $destination = String::ReplaceAll({out_path!r}, "X_ID", $id);

                INSERT INTO $destination WITH TRUNCATE
                SELECT * FROM $source
                WHERE date_end > {ttl!r}
                ORDER BY id_type, id
                ;
            END DEFINE;

            EVALUATE FOR $id IN AsList({id_types})
                DO $fresh($id)
            ;
        """

        with self.yt.Transaction() as tx:
            ttl = utils.get_date_before(self.date, int(config.STORE_DAYS))
            id_types = ", ".join(map(lambda item: repr(item.Name), self.id_types))
            in_path = ypj(self.out_f("ids_storage"), "X_ID", "eternal")
            out_path = ypj(self.out_f("ids_storage"), "X_ID", "fresh")
            run_yql(query=template.format(**locals()))

            for idt in self.id_types:
                out_path = idstorage_path(self.out_f("ids_storage"), idt, "fresh")
                mr.set_generate_date(out_path, self.date)

    def output(self):
        ids_storage_f = self.out_f("ids_storage")
        return [
            yt_luigi.YtDateTarget(idstorage_path(ids_storage_f, id_type, "fresh"), self.date)
            for id_type in self.id_types
        ]
