from textwrap import dedent

import luigi
import yt.wrapper as yt
from crypta.graph.soup.config.python import (  # N811 # noqa
    ID_TYPE as ids,
    LOG_SOURCE as log_source,
    SOURCE_TYPE as source_type,
    EDGE_TYPE as edge_type,
)

from crypta.graph.v1.python.lib.luigi import yt_luigi
from crypta.graph.v1.python.rtcconf import config
from crypta.graph.v1.python.utils import mr_utils as mr
from crypta.graph.v1.python.utils.yql_utils import run_yql
from crypta.graph.v1.python.v2.ids_storage.eternal import UpdateIdStorageApps
from crypta.graph.v1.python.v2.soup.graph_soup_update import AddLogToSoup
from crypta.graph.v1.python.v2.soup.soup_tables import SoupStorageTable

YQL = dedent(
    """
    $uuid_info = '{uuid_info_table}';
    $mm_device_id_uuid_soup =  '{mm_device_id_uuid_soup_table}';

    $uuid_type = '{uuid_type}';
    $mm_device_id_type = '{mm_device_id_type}';
    $source = '{source}';
    $yandex_drive_ids = AsList({yandex_drive_ids});

    $output = '{output}';
    $public_drive_uuid = '{public_drive_uuid}';
    $public_drive_mm_device_id = '{public_drive_mm_device_id}';

    $uuid_from_yandex_drive = (
        SELECT
            id,
            $uuid_type AS id_type,
            $source AS source,
            date_begin,
            date_end
        FROM $uuid_info
        WHERE app_id IN $yandex_drive_ids
        -- already unique
    );

    $mmdevice_from_yandex_drive = (
        SELECT
            id,
            $mm_device_id_type as id_type,
            $source as source,
            MAX(u.date_begin) as date_begin,
            MIN(u.date_end) as date_end
        FROM $uuid_from_yandex_drive as u
        INNER JOIN ANY $mm_device_id_uuid_soup as soup
        ON u.id == soup.id2
        -- make unique
        GROUP BY soup.id1 AS id
    );

    -- publish

    INSERT INTO $public_drive_uuid WITH TRUNCATE
    SELECT *
    FROM $uuid_from_yandex_drive
    ORDER BY id, id_type;

    INSERT INTO $public_drive_mm_device_id WITH TRUNCATE
    SELECT *
    FROM $mmdevice_from_yandex_drive
    ORDER BY id, id_type;

    INSERT INTO $output WITH TRUNCATE
    SELECT * FROM (
        SELECT * FROM $uuid_from_yandex_drive
        UNION ALL
        SELECT * FROM $mmdevice_from_yandex_drive
    ) ORDER BY id, id_type;
    """
)


class YandexDrive(yt_luigi.BaseYtTask):
    date = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(YandexDrive, self).__init__(*args, **kwargs)
        self.yandex_drive_ids = ["com.yandex.mobile.auto", "yandex.auto", "com.yandex.drive.agent"]
        # for some reason is used in dependent tasks
        self.output_folder_name = "yandex_drive"
        # for some reason is used in dependent tasks
        self.source = "Yandex Drive"

        self.mm_device_id_uuid_soup = edge_type.get_edge_type(
            ids.MM_DEVICE_ID, ids.UUID, source_type.APP_METRICA, log_source.METRIKA_MOBILE_LOG
        )

    def requires(self):
        return [
            UpdateIdStorageApps(self.date),
            AddLogToSoup(date=self.date, log_source_type=log_source.METRIKA_MOBILE_LOG.Type),
        ]

    def input_folders(self):
        return {"ids_storage": config.CRYPTA_IDS_STORAGE}

    def output_folders(self):
        return {"shared_ids": config.CRYPTA_SHARED_IDS_FOLDER, "ids_storage": config.CRYPTA_IDS_STORAGE}

    def output(self):
        return [
            yt_luigi.YtDateTarget(t, self.date)
            for t in [self.output_path, self.ids_storage_uuid, self.ids_storage_mm_device_id]
        ]

    @property
    def output_path(self):
        return yt.ypath_join(self.out_f("shared_ids"), self.output_folder_name, self.date)

    @property
    def ids_storage_uuid(self):
        return yt.ypath_join(self.out_f("ids_storage"), ids.UUID.Name, "yandex_drive")

    @property
    def ids_storage_mm_device_id(self):
        return yt.ypath_join(self.out_f("ids_storage"), ids.MM_DEVICE_ID.Name, "yandex_drive")

    @staticmethod
    def desc():
        # for some reason is used in dependent tasks
        return "YANDEX_DRIVE"

    def run(self):
        query = YQL.format(
            uuid_info_table=yt.ypath_join(self.in_f("ids_storage"), ids.UUID.Name, "eternal"),
            mm_device_id_uuid_soup_table=SoupStorageTable(self.mm_device_id_uuid_soup, self.date).table_path(),
            uuid_type=ids.UUID.Name,
            mm_device_id_type=ids.MM_DEVICE_ID.Name,
            yandex_drive_ids=", ".join(map(repr, self.yandex_drive_ids)),
            source="Yandex Drive",
            output=self.output_path,
            public_drive_uuid=self.ids_storage_uuid,
            public_drive_mm_device_id=self.ids_storage_mm_device_id,
        )
        run_yql(query=query)

        for t in [self.output_path, self.ids_storage_uuid, self.ids_storage_mm_device_id]:
            mr.set_generate_date(t, self.date)
