from __future__ import print_function
from collections import namedtuple
from datetime import datetime
import os
import luigi
from yt.wrapper import YtClient
from yt.transfer_manager.client import TransferManager

from crypta.graph.v1.python.v2.v2_task_status import SoupIsReadyFast
from crypta.graph.v1.python.lib.luigi import yt_luigi
from crypta.graph.v1.python.utils import yt_clients
from crypta.graph.v1.python.v2.soup.soup_dirs import SOUP_DIR
from crypta.graph.v1.python.rtcconf import config

from crypta.graph.soup.config.python import EDGE_TYPE, LOG_SOURCE, ID_TYPE


BackupVariant = namedtuple("BackupVariant", "name is_applicable keep_count")


def get_backup_edge_types():
    return [
        x
        for x in EDGE_TYPE.values()
        if x.LogSource.Type
        not in [
            LOG_SOURCE.SOUP_PREPROCESSING.Type,
            LOG_SOURCE.CRYPTA_BAYES.Type,
            LOG_SOURCE.CRYPTA_INDEVICE.Type,
            # LOG_SOURCE.FUZZY2_INDEVICE.Type,
            LOG_SOURCE.HEURISTIC.Type,
        ]
    ]


def do_copy(yt_client, src_dir, dst_dir):
    if not yt_client.exists(dst_dir):
        yt_client.mkdir(dst_dir, recursive=True)

    for et in get_backup_edge_types():
        table_name = EDGE_TYPE.name(et)
        yt_client.copy(os.path.join(src_dir, table_name), os.path.join(dst_dir, table_name))


def do_cleanup(ytclient, bkdir, keep_count):
    tables = sorted(ytclient.list(bkdir, absolute=True), reverse=True)
    to_remove = tables[keep_count:]
    for t in to_remove:
        ytclient.remove(t, recursive=True)


def remote_backup(master, slave, backup_dir, variants):
    yt_client = yt_clients.get_yt_client()
    soup_date = yt_client.get_attribute(SOUP_DIR[:-1], "generate_date")
    soup_dt = datetime.strptime(soup_date, "%Y-%m-%d")

    if master != slave:
        tm = TransferManager()
        remote_yt = YtClient(proxy=slave, token=os.getenv("YT_TOKEN"))
    else:
        tm = None
        remote_yt = yt_client

    applicables = [v for v in variants if v.is_applicable(soup_dt) and v.keep_count > 0]
    if applicables:
        src_dst = []
        remote_dir = os.path.join(backup_dir, applicables[0].name, soup_date)

        for et in get_backup_edge_types():
            table = EDGE_TYPE.name(et)
            src_dst.append((os.path.join(SOUP_DIR, table), os.path.join(remote_dir, table)))

        if config.ENABLE_IDSTORAGE_BACKUP:
            idstorage_idtypes = [
                ID_TYPE.YANDEXUID,
                ID_TYPE.ICOOKIE,
                ID_TYPE.MM_DEVICE_ID,
                ID_TYPE.IDFA,
                ID_TYPE.GAID,
                ID_TYPE.UUID,
            ]
            for idt in idstorage_idtypes:
                src = os.path.join(config.CRYPTA_IDS_STORAGE, idt.Name, "eternal")
                dst = os.path.join(remote_dir, "ids_storage", idt.Name, "eternal")
                src_dst.append((src, dst))

        if tm is not None:
            tm.add_tasks_from_src_dst_pairs(src_dst, master, slave, sync=True, running_tasks_limit=20)
        else:
            for src, dst in src_dst:
                remote_yt.create("map_node", os.path.dirname(dst), recursive=True, ignore_existing=True)
                print("Copy {} to {}".format(src, dst))
                remote_yt.copy(src, dst, force=True, recursive=True)
        do_cleanup(remote_yt, os.path.join(backup_dir, applicables[0].name), applicables[0].keep_count)

    for v in applicables[1:]:
        src_dir = os.path.join(backup_dir, applicables[0].name, soup_date)
        dst_dir = os.path.join(backup_dir, v.name, soup_date)
        do_copy(remote_yt, src_dir, dst_dir)
        do_cleanup(remote_yt, os.path.join(backup_dir, v.name), v.keep_count)


def backup_soup():
    remote_backup(
        config.SOUP_BACKUP_MASTER,
        config.SOUP_BACKUP_SLAVE,
        config.SOUP_BACKUP_FOLDER,
        [
            BackupVariant("day", lambda dt: True, config.SOUP_BACKUP_KEEP_DAYS),
            BackupVariant("week", lambda dt: dt.weekday() == 6, config.SOUP_BACKUP_KEEP_WEEKS),  # on sundays
            BackupVariant("month", lambda dt: dt.day == 1, config.SOUP_BACKUP_KEEP_MONTHS),
        ],
    )


class SoupBackupTask(yt_luigi.BaseYtTask):
    date = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(SoupBackupTask, self).__init__(*args, **kwargs)
        self.target = yt_luigi.YtDateAttributeTarget(SOUP_DIR.rstrip("/"), "backed_up", self.date)

    def requires(self):
        return [SoupIsReadyFast(self.date)]

    def output(self):
        return self.target

    def run(self):
        backup_soup()
        self.target.set_date(self.date)
