from __future__ import print_function

import os
from os.path import join as ypj

from crypta.lib.python.yql_runner.base_parser import BaseParser
from crypta.graph.soup.config.python import (
    ID_TYPE as id_type,
    SOURCE_TYPE as source_type,
    LOG_SOURCE as log_source,
    EDGE_TYPE as edges,
)
from crypta.graph.v1.python.rtcconf import config

is_embedded = (config.CRYPTA_ENV == "development") or config.YQL_FORCE_EMBEDDED


def edge_table_name(*args):
    return edges.name(edges.get_edge_type(*args))


def normalize_path(path):
    if path and path.endswith("/"):
        return path[:-1]
    else:
        return path


class UniqueIdsYqlQuery(BaseParser):
    QUERY_TEMPLATE = "unique_ids.sql.j2"

    def __init__(self, yt_proxy, tx, date, soup_dir, out_dir, edge_types):
        self.tx = tx
        self.soup_dir = soup_dir
        self.out_dir = out_dir
        self.edge_types = edge_types
        super(UniqueIdsYqlQuery, self).__init__(
            date=date,
            yt_proxy=yt_proxy,
            pool=os.getenv("YT_POOL"),
            mrjob_binary=config.YQL_MRJOB,
            udf_resolver_binary=config.YQL_UDF_RESOLVER,
            udfs_dir=config.YQL_UDF_DIR,
            is_embedded=is_embedded,
        )

    def get_context_data(self, **kwargs):
        edge_types_list = [edges.name(et) for et in self.edge_types]
        id_types_list = [t.Name for t in id_type.values()]

        """ Fill context data for template rendering """
        context = super(UniqueIdsYqlQuery, self).get_context_data(**kwargs)
        context.update(tx=self.tx, edge_types_list=edge_types_list, id_types_list=id_types_list)
        return context

    def get_dirs(self):
        """Get current environ and replace directories path to env"""
        dirs = super(UniqueIdsYqlQuery, self).get_dirs()
        dirs.update(soup_dir=self.soup_dir, out_dir=self.out_dir)
        return dirs


class SoupCookYqlQuery(BaseParser):
    QUERY_TEMPLATE = "soup_cook.sql.j2"

    def __init__(self, yt_proxy, tx, date, soup_dir, ids_dir, out_soup_dir, out_ids_dir, uuid_eternal_tbl):
        self.tx = tx
        self.soup_dir = soup_dir
        self.ids_dir = ids_dir
        self.out_soup_dir = out_soup_dir
        self.out_ids_dir = out_ids_dir
        self.uuid_eternal_tbl = uuid_eternal_tbl

        super(SoupCookYqlQuery, self).__init__(
            date=date,
            yt_proxy=yt_proxy,
            pool=os.getenv("YT_POOL"),
            mrjob_binary=config.YQL_MRJOB,
            udf_resolver_binary=config.YQL_UDF_RESOLVER,
            udfs_dir=config.YQL_UDF_DIR,
            is_embedded=is_embedded,
        )

    def get_context_data(self, **kwargs):
        """Fill context data for template rendering"""
        context = super(SoupCookYqlQuery, self).get_context_data(**kwargs)
        context.update(
            tx=self.tx,
            avito_salt="fy5drs34dgh13ff",
            uuid_eternal_tbl=self.uuid_eternal_tbl,
            out_tables={
                ("login", "email"): edge_table_name(
                    id_type.LOGIN, id_type.EMAIL, source_type.LOGIN_TO_EMAIL, log_source.SOUP_PREPROCESSING
                ),
                ("email", "phone"): edge_table_name(
                    id_type.EMAIL, id_type.PHONE, source_type.EMAIL_TO_PHONE, log_source.SOUP_PREPROCESSING
                ),
                ("email", "md5"): edge_table_name(
                    id_type.EMAIL, id_type.EMAIL_MD5, source_type.MD5_HASH, log_source.SOUP_PREPROCESSING
                ),
                ("email", "sha256"): edge_table_name(
                    id_type.EMAIL, id_type.EMAIL_SHA256, source_type.SHA256_HASH, log_source.SOUP_PREPROCESSING
                ),
                ("email", "md5", "sha256"): edge_table_name(
                    id_type.EMAIL_MD5, id_type.EMAIL_SHA256, source_type.HASH_TO_HASH, log_source.SOUP_PREPROCESSING
                ),
                ("phone", "md5"): edge_table_name(
                    id_type.PHONE, id_type.PHONE_MD5, source_type.MD5_HASH, log_source.SOUP_PREPROCESSING
                ),
                ("phone", "sha256"): edge_table_name(
                    id_type.PHONE, id_type.PHONE_SHA256, source_type.SHA256_HASH, log_source.SOUP_PREPROCESSING
                ),
                ("phone", "md5", "sha256"): edge_table_name(
                    id_type.PHONE_MD5, id_type.PHONE_SHA256, source_type.HASH_TO_HASH, log_source.SOUP_PREPROCESSING
                ),
                ("email", "avito_hash"): edge_table_name(
                    id_type.EMAIL, id_type.AVITO_HASH, source_type.AVITO, log_source.SOUP_PREPROCESSING
                ),
                ("yandexuid", "email"): edge_table_name(
                    id_type.YANDEXUID, id_type.EMAIL, source_type.EMAIL_SENDER, log_source.SOUP_PREPROCESSING
                ),
                ("uuid", "icookie"): edge_table_name(
                    id_type.UUID, id_type.ICOOKIE, source_type.ICOOKIE_HASH, log_source.SOUP_PREPROCESSING
                ),
            },
        )
        return context

    def get_dirs(self):
        """Get current environ and replace directories path to env"""
        dirs = super(SoupCookYqlQuery, self).get_dirs()
        dirs.update(
            soup_dir=self.soup_dir,
            ids_dir=self.ids_dir,
            out_ids_dir=self.out_ids_dir,
            out_soup_dir=self.out_soup_dir,
        )

        return dirs


def cook_soup_yql(yt_proxy, tx, date, in_soup_dir, out_ids_dir, out_soup_dir, soup_edges_before_preproc):
    in_soup_dir = normalize_path(in_soup_dir)
    out_ids_dir = normalize_path(out_ids_dir)
    out_soup_dir = normalize_path(out_soup_dir)
    uuid_eternal_tbl = ypj(config.CRYPTA_IDS_STORAGE, id_type.UUID.Name, "eternal")

    yql1 = UniqueIdsYqlQuery(
        yt_proxy=yt_proxy,
        tx=tx.transaction_id,
        date=date,
        soup_dir=in_soup_dir,
        out_dir=out_ids_dir,
        edge_types=soup_edges_before_preproc,
    )

    yql1.run()

    yql2 = SoupCookYqlQuery(
        yt_proxy=yt_proxy,
        tx=tx.transaction_id,
        date=date,
        soup_dir=in_soup_dir,
        ids_dir=out_ids_dir,
        out_ids_dir=out_ids_dir,
        out_soup_dir=out_soup_dir,
        uuid_eternal_tbl=uuid_eternal_tbl,
    )

    yql2.run()
