import datetime

import luigi
from crypta.graph.v1.python.lib.luigi import yt_luigi
from crypta.graph.v1.python.rtcconf import config

from crypta.graph.v1.python.utils import utils
from crypta.graph.v1.python.utils import mr_utils as mr
from crypta.graph.v1.python.utils.yql_utils import run_yql
from crypta.graph.v1.python.data_imports.import_logs.app_metrica_day import ImportAppMetrikaDayTask
from crypta.graph.v1.python.matching.yuid_matching import graph_yuid_info

from crypta.graph.v1.python.v2.soup.soup_tables import SoupDumpTable

from crypta.graph.soup.config.python import (  # noqa
    ID_TYPE as ids,
    SOURCE_TYPE as source_type,
    LOG_SOURCE as log_source,
    EDGE_TYPE as edges,
)


class Fuzzy2DayTask(yt_luigi.BaseYtTask):

    date = luigi.Parameter()
    run_date = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(Fuzzy2DayTask, self).__init__(*args, **kwargs)

    def output_folders(self):
        return {"fuzzy": config.INDEVICE_YT_FOLDER + self.date + "/fuzzy/"}

    def requires(self):
        return [
            ImportAppMetrikaDayTask(date=self.date, run_date=self.run_date),
            yt_luigi.ExternalInput(config.LOG_FOLDERS["bs_watch"] + self.date),
        ]

    def run(self):
        run_yql(
            "Fuzzy2Parser",
            dict(fuzzy2_use_un=config.FUZZY2_USE_UN, is_day_task=True, date_start=self.date, date_end=self.date),
        )

        mr.drop(self.out_f("fuzzy") + "fuzzy2_metrica")
        mr.drop(self.out_f("fuzzy") + "fuzzy2_bswatch")

    def output(self):
        return [yt_luigi.YtTarget(t) for t in [self.out_f("fuzzy") + "dev_yuid_fuzzy2"]]


class Fuzzy2MonthTask(yt_luigi.BaseYtTask):

    date = luigi.Parameter()
    resources = {"fuzzy2_dump_lock": 1}

    def __init__(self, *args, **kwargs):
        super(Fuzzy2MonthTask, self).__init__(*args, **kwargs)
        self.soup_storages = [
            SoupDumpTable(
                edges.get_edge_type(ids.IDFA, ids.YANDEXUID, source_type.PROBABILISTIC2, log_source.FUZZY2_INDEVICE),
                self.date,
            ),
            SoupDumpTable(
                edges.get_edge_type(ids.GAID, ids.YANDEXUID, source_type.PROBABILISTIC2, log_source.FUZZY2_INDEVICE),
                self.date,
            ),
        ]
        self.date_start = (
            datetime.datetime.strptime(self.date, "%Y-%m-%d") - datetime.timedelta(days=int(config.STORE_DAYS))
        ).strftime("%Y-%m-%d")

    def output_folders(self):
        return {"fuzzy": config.INDEVICE_YT_FOLDER + self.date + "/fuzzy/"}

    def requires(self):
        return [graph_yuid_info.YuidInfoMonth(date=self.date)] + [
            Fuzzy2DayTask(date=day, run_date=self.date)
            for day in utils.get_dates_before(self.date, int(config.STORE_DAYS))
        ]

    def run(self):
        run_yql(
            "Fuzzy2Parser",
            dict(
                fuzzy2_use_un=config.FUZZY2_USE_UN, is_day_task=False, date_start=self.date_start, date_end=self.date
            ),
        )
        for soup in self.soup_storages:
            soup.finalize()

    def output(self):
        return [yt_luigi.YtTarget(t) for t in [self.out_f("fuzzy") + "dev_yuid_unperfect"]] + [
            soup.as_target() for soup in self.soup_storages
        ]
