import sys

from collections import deque

import luigi
import yt.wrapper as yt

from crypta.graph.v1.python.v2.soup import soup_dirs
from crypta.graph.v1.python.utils import yt_clients


class TableRegister(yt_clients.YtClientMixin):
    def __init__(self):
        self.input_tables = []
        self.output_tables = []
        self.error_tables = []

    @staticmethod
    def output_tables_name():
        return "soup_output_tables"

    @staticmethod
    def input_tables_name():
        return "soup_input_tables"

    @staticmethod
    def error_tables_name():
        return "soup_error_tables"

    @property
    def attribute_name(self):
        return "created_by_task"

    def register_output_table(self, table, task):
        record = {"table_path": table, "task": task}
        if record not in self.output_tables:
            self.output_tables.append(record)
        if self.yt.exists(table):
            try:
                self.yt.set_attribute(table, self.attribute_name, task)
            except yt.errors.YtHttpResponseError:
                self.error_tables.append(dict(table_path=table, exception=str(sys.exc_info()[1])))

    def register_output_tables(self, output_tables, output_duplicated_table):
        for table, task in output_tables.items():
            self.register_output_table(table, task)
        for table in output_duplicated_table:
            record = {"table_path": table["table"], "task": table["task"]}
            if record not in self.output_tables:
                self.output_tables.append(record)

    def register_input_table(self, table, task):
        record = {"table_path": table, "task": task}
        if record not in self.input_tables:
            self.input_tables.append(record)

    def register_input_tables(self, tables):
        for table in tables:
            self.register_input_table(table["table"], table["task"])

    def get_output_tables(self):
        return self.output_tables

    def get_input_tables(self):
        return self.input_tables

    def get_error_tables(self):
        return self.error_tables

    def write_tables_to_yt(self):
        def attributes():
            attribs = {
                "schema": [{"name": "table_path", "type": "string"}, {"name": "task", "type": "string"}],
                self.attribute_name: self.__class__.__name__,
            }
            return attribs

        def error_attributes():
            attribs = {
                "schema": [{"name": "table_path", "type": "string"}, {"name": "exception", "type": "string"}],
                self.attribute_name: self.__class__.__name__,
            }
            return attribs

        output_table = yt.ypath_join(soup_dirs.V2_DIR, self.output_tables_name())
        input_table = yt.ypath_join(soup_dirs.V2_DIR, self.input_tables_name())
        error_table = yt.ypath_join(soup_dirs.V2_DIR, self.error_tables_name())
        with self.yt.Transaction():
            self.yt.create("table", output_table + "_by_task", recursive=True, force=True, attributes=attributes())
            if self.output_tables:
                self.yt.write_table(output_table + "_by_task", self.get_output_tables())
                self.yt.run_sort(output_table + "_by_task", sort_by=["task", "table_path"])
            self.yt.create(
                "table", output_table + "_by_table_path", recursive=True, force=True, attributes=attributes()
            )
            if self.output_tables:
                self.yt.write_table(output_table + "_by_table_path", self.get_output_tables())
                self.yt.run_sort(output_table + "_by_table_path", sort_by=["table_path", "task"])
            self.yt.create("table", input_table + "_by_task", recursive=True, force=True, attributes=attributes())
            if self.input_tables:
                self.yt.write_table(input_table + "_by_task", self.get_input_tables())
                self.yt.run_sort(input_table + "_by_task", sort_by=["task", "table_path"])
            self.yt.create(
                "table", input_table + "_by_table_path", recursive=True, force=True, attributes=attributes()
            )
            if self.input_tables:
                self.yt.write_table(input_table + "_by_table_path", self.get_input_tables())
                self.yt.run_sort(input_table + "_by_table_path", sort_by=["table_path", "task"])
            self.yt.create("table", error_table, recursive=True, force=True, attributes=error_attributes())
            if self.error_tables:
                self.yt.write_table(error_table, self.get_error_tables())
                self.yt.run_sort(error_table, sort_by=["table_path", "exception"])


class FlattenTasks(object):
    def __init__(self, object_ref):
        self.deque = deque()
        self.deque.append(object_ref)
        self.input_tables = []
        self.output_tables = {}
        self.output_duplicated_tables = []

    def mark_tables(self):
        self._flatten()
        registry = TableRegister()
        if self.output_tables:
            registry.register_output_tables(self.output_tables, self.output_duplicated_tables)
        if self.input_tables:
            registry.register_input_tables(self.input_tables)
        registry.write_tables_to_yt()

    def _get_output_tables(self):
        self._flatten()
        return self.output_tables

    def _flatten(self):
        while len(self.deque):
            cur_object = self.deque.popleft()
            if not isinstance(cur_object, object):
                continue
            if hasattr(cur_object, "output") and callable(cur_object.output):
                tasks = cur_object.output() if isinstance(cur_object.output(), list) else [cur_object.output()]
                for task in tasks:
                    if isinstance(task, luigi.Target):
                        if hasattr(task, "table") and not callable(task.table):
                            if task.table not in self.output_tables.keys():
                                self.output_tables[task.table] = cur_object.__class__.__name__
                            elif self.output_tables[task.table] != cur_object.__class__.__name__:
                                self.output_duplicated_tables.append(
                                    dict(table=task.table, task=cur_object.__class__.__name__)
                                )

            if hasattr(cur_object, "requires") and callable(cur_object.requires):
                tasks = cur_object.requires() if isinstance(cur_object.requires(), list) else [cur_object.requires()]
                for task in tasks:
                    if isinstance(task, luigi.ExternalTask):
                        if hasattr(task, "table") and not callable(task.table):
                            self.input_tables.append(dict(table=task.table, task=cur_object.__class__.__name__))
                    else:
                        self.deque.append(task)
