import luigi
import yt.wrapper as yt

from textwrap import dedent

import crypta.graph.v1.python.utils.mr_utils as mr
from crypta.graph.v1.python.lib.luigi import yt_luigi
from crypta.graph.v1.python.matching.device_matching.app_metrica.app_metrica_month import AppMetricaDictMergeMonthTask
from crypta.graph.v1.python.matching.yuid_matching.graph_yuid_info import YuidInfoMonth
from crypta.graph.v1.python.rtcconf import config
from crypta.graph.v1.python.utils.yql_utils import run_yql
from crypta.graph.v1.python.v2.shared.heuristic_desktop_yuids import HeuristicSharedDesktopYuids
from crypta.graph.v1.python.v2.shared.yandex_drive import YandexDrive
from crypta.graph.v1.python.v2.soup.graph_soup_cook import SoupPreprocessing
from crypta.graph.v1.python.v2.soup.graph_soup_update import AddDayToSoup, AddDumpsToSoup, AddFuzzyToSoup, AddLogToSoup

from crypta.graph.soup.config.python import (  # N811 # noqa
    LOG_SOURCE as log_source,
)
from crypta.graph.soupy_indevice.lib import indevice_edge_types
from crypta.graph.v1.python.v2.ids_storage.eternal import UpdateEternalIdStorage

from crypta.graph.v1.python.v2.soup.soup_dirs import SOUP_DIR

YQL = dedent(
    """
    $output = '{output}';

    $output_desc = '{output_desc}';

    $source_desc = AsDict(
        {source_desc}
    );

    INSERT INTO $output WITH TRUNCATE
    SELECT id, id_type, AGGREGATE_LIST_DISTINCT(source) AS source
    FROM CONCAT({shared_tables})
    GROUP BY id, id_type
    ORDER BY id, id_type;

    INSERT INTO $output_desc WITH TRUNCATE
    SELECT id, id_type, AGGREGATE_LIST_DISTINCT($source_desc[source]) AS shared_types
    FROM CONCAT({shared_tables})
    GROUP BY id, id_type
    ORDER BY id, id_type;
"""
)


class MergeSharedVertices(yt_luigi.BaseYtTask):
    date = luigi.Parameter()

    def requires(self):
        return [YandexDrive(self.date), HeuristicSharedDesktopYuids(self.date)]

    def input_folders(self):
        return {"shared_ids": config.CRYPTA_SHARED_IDS_FOLDER, "ids_storage": config.CRYPTA_IDS_STORAGE}

    def output_folders(self):
        return {"shared_merged": yt.ypath_join(self.in_f("shared_ids"), "merged")}

    @property
    def output_desc(self):
        return yt.ypath_join(self.in_f("ids_storage"), "shared", "common_shared")

    @property
    def output_path(self):
        return yt.ypath_join(self.out_f("shared_merged"), self.date)

    def output(self):
        return [yt_luigi.YtDateTarget(self.output_path, self.date), yt_luigi.YtDateTarget(self.output_desc, self.date)]

    def shared_tables(self):
        shared_folders = [
            yt.ypath_join(self.in_f("shared_ids"), shared_class.output_folder_name) for shared_class in self.requires()
        ]
        shared_tables = []
        for folder in shared_folders:
            if self.yt.exists(folder):
                shared_tables.extend(
                    [table for table in self.yt.list(folder, absolute=True) if table.split("/")[-1] == self.date]
                )
        shared_tables = list(set(shared_tables))
        return shared_tables

    def source_desc(self):
        desc = []
        for req in self.requires():
            desc.append('AsTuple("{source}", "{desc}")'.format(source=req.source, desc=req.desc()))
        return ",\n".join(desc)

    def run(self):
        query = YQL.format(
            shared_tables=",\n".join(("'{}'".format(table) for table in self.shared_tables())),
            output=self.output_path,
            date=self.date,
            source_desc=self.source_desc(),
            output_desc=self.output_desc,
        )
        run_yql(query=query)

        mr.set_generate_date(self.output_path, self.date)
        mr.set_generate_date(self.output_desc, self.date)


class IdsStorageIsReady(luigi.WrapperTask):
    date = luigi.Parameter()

    def requires(self):
        return [YuidInfoMonth(self.date), AppMetricaDictMergeMonthTask(self.date)]

    def run(self):
        mr.set_generate_date(config.CRYPTA_IDS_STORAGE, self.date)


class SoupIsReadyFast(luigi.WrapperTask):
    """
    Don't want to wait for fuzzy
    """

    date = luigi.Parameter()
    priority = 2

    def requires(self):
        return [
            AddDayToSoup(self.date),
            AddDumpsToSoup(self.date, active_dump=False),
            AddDumpsToSoup(self.date, active_dump=True),
            SoupPreprocessing(self.date),
        ]

    def run(self):
        mr.set_generate_date(SOUP_DIR, self.date)


class SoupIsReadyAll(luigi.WrapperTask):
    date = luigi.Parameter()

    def requires(self):
        return [SoupIsReadyFast(self.date), AddFuzzyToSoup(self.date)]


class V2IsReadyFast(luigi.WrapperTask):
    """
    Don't want to wait for fuzzy
    """

    date = luigi.Parameter()
    priority = 1

    def requires(self):
        return [SoupIsReadyFast(self.date), IdsStorageIsReady(self.date), MergeSharedVertices(self.date)]


class V2IsReadyAll(luigi.WrapperTask):
    date = luigi.Parameter()

    def requires(self):
        return [SoupIsReadyAll(self.date), IdsStorageIsReady(self.date)]


class IndeviceStuffIsReadyFast(luigi.WrapperTask):
    date = luigi.Parameter()
    priority = 1

    def requires(self):
        indevice_logs = set(
            [
                et.LogSource.Type
                for et in indevice_edge_types()
                if et.Usage.SoupUpdate
                # no wait till postprocess
                and et.LogSource != log_source.SOUP_PREPROCESSING
            ]
        )

        return [AddLogToSoup(date=self.date, log_source_type=x) for x in indevice_logs]


class IndeviceStuffIsReady(luigi.WrapperTask):
    date = luigi.Parameter()
    priority = 1

    def requires(self):
        return [UpdateEternalIdStorage(self.date), IndeviceStuffIsReadyFast(self.date)]
