from __future__ import print_function

import os
import datetime

from cached_property import cached_property

import crypta.lib.python.bt.conf.conf as conf  # noqa

from crypta.lib.python.bt.workflow import Parameter
from crypta.lib.python.yql_runner.task import YQLRunnerTask


def date_or_none(tbl):
    try:
        name = str(tbl).split("/")[-1]
        return datetime.datetime.strptime(name, "%Y-%m-%d")
    except ValueError:
        return None


class VavilovTask(YQLRunnerTask):
    """Run Vavilov to make ids diff"""

    enabled_ids = Parameter(
        parse=lambda value: [each.strip() for each in value.split(",") if each.strip()], default=""
    )
    keep = Parameter(parse=int, default=3)

    @property
    def query_template(self):
        return "vavilov.sql.j2"

    def requires(self, *args, **kwargs):
        return []

    def run(self, *args, **kwargs):
        self.prepare()
        super(VavilovTask, self).run(*args, **kwargs)
        self._set_generate_date()
        self.cleanup()

    def _set_generate_date(self):
        for path in (self.diff_state_path, self.cid_history_path, self.cid_history_export_path):
            self.yt.set("{path}/@generate_date".format(path=path), self.date)

    def prepare(self):
        if not self.yt.exists(self.base_path):
            self.yt.create("map_node", self.base_path, recursive=True)
        if self.previous_state_path == self.current_state_path:
            raise Exception("Duplicate date run for '{}'".format(self.date))

    def cleanup(self):
        """clean up old tables"""
        for tbl in self.list_tables()[self.keep :]:
            print("Remove {}".format(tbl))
            self.yt.remove(tbl, recursive=True)

    @cached_property
    def vertices_by_type_path(self):
        return "//home/crypta/{crypta_env}/state/graph/v2/matching/vertices_no_multi_profile_by_id_type".format(
            crypta_env=self.crypta_env
        )

    @cached_property
    def vertices_path(self):
        return "//home/crypta/{crypta_env}/state/graph/v2/matching/vertices_no_multi_profile".format(
            crypta_env=self.crypta_env
        )

    @cached_property
    def cid_history_path(self):
        return "//home/crypta/{crypta_env}/state/graph/v2/matching/vertices_history".format(crypta_env=self.crypta_env)

    @cached_property
    def cid_history_export_path(self):
        return "//home/crypta/{crypta_env}/state/graph/profiles/vertices_history".format(crypta_env=self.crypta_env)

    @cached_property
    def base_path(self):
        return "//home/crypta/{crypta_env}/state/graph/v2/vavilov".format(crypta_env=self.crypta_env)

    @cached_property
    def profile_dump_path(self):
        return max(map(str, self.yt.list("//home/bigb/production/public/profiles", absolute=True)))

    @cached_property
    def current_state_path(self):
        return os.path.join(self.base_path, self.date)

    @cached_property
    def previous_state_path(self):
        tables = self.list_tables()
        if len(tables) == 0:
            # no any tables in prev state
            return None
        return tables[0]

    @cached_property
    def diff_state_path(self):
        return os.path.join(self.base_path, "diff")

    @cached_property
    def date(self):
        return self.yt.get(os.path.join(self.vertices_path, "@generate_date"))

    def list_tables(self):
        return sorted(
            filter(
                date_or_none,
                self.yt.list(self.base_path, absolute=True),
            ),
            key=date_or_none,
            reverse=True,
        )

    def get_context_data(self, **kwargs):
        context = super(VavilovTask, self).get_context_data(**kwargs)
        context.update(
            generate_date=self.date,
            diff_state=self.diff_state_path,
            current_state=self.current_state_path,
            previous_state=self.previous_state_path,
            vertices_by_id=self.vertices_path,
            vertices_by_type=self.vertices_by_type_path,
            cid_history=self.cid_history_path,
            cid_history_export=self.cid_history_export_path,
            profile_dump=self.profile_dump_path,
            enabled_id_types=self.enabled_ids,
        )
        return context
