package ru.yandex.crypta.idm;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.annotation.JsonProperty;

public class RoleDefinition {

    private final String slug;
    private final Map<String, RoleDefinition> values;
    private final RoleDefinition roles;

    private final LocalizedString name;
    private final LocalizedString help;

    private List<FieldDefinition> fields;

    @JsonProperty("comment_required")
    private boolean commentRequired;

    public RoleDefinition(String slug, Map<String, RoleDefinition> values,
            RoleDefinition roles, LocalizedString name,
            LocalizedString help, List<FieldDefinition> fields, boolean commentRequired)
    {
        this.slug = slug;
        this.values = values;
        this.roles = roles;
        this.name = name;
        this.help = help;
        this.fields = fields;
        this.commentRequired = commentRequired;
    }

    public String getSlug() {
        return slug;
    }

    public Map<String, RoleDefinition> getValues() {
        return values;
    }

    public RoleDefinition getRoles() {
        return roles;
    }

    public LocalizedString getName() {
        return name;
    }

    public LocalizedString getHelp() {
        return help;
    }

    public List<FieldDefinition> getFields() {
        return fields;
    }

    public boolean isCommentRequired() {
        return commentRequired;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {

        private String slug;
        private LocalizedString name;
        private Map<String, RoleDefinition> values;
        private LocalizedString help;
        private RoleDefinition roles;
        private List<FieldDefinition> fields;
        private boolean commentRequired = true;

        private Builder setSlug(String slug) {
            this.slug = slug;
            return this;
        }

        public Builder group() {
            return setSlug("group");
        }

        public Builder subgroup() {
            return setSlug("subgroup");
        }

        public Builder subsubgroup() {
            return setSlug("subsubgroup");
        }

        public Builder setName(LocalizedString.Builder name) {
            this.name = name.build();
            return this;
        }

        public Builder setName(LocalizedString name) {
            this.name = name;
            return this;
        }

        public Builder value(String key, RoleDefinition value) {
            if (this.values == null) {
                this.values = new HashMap<>();
            }
            this.values.put(key, value);
            return this;
        }

        public Builder setHelp(LocalizedString.Builder help) {
            this.help = help.build();
            return this;
        }

        public Builder setRoles(RoleDefinition.Builder roles) {
            this.roles = roles.build();
            return this;
        }

        public Builder setRoles(RoleDefinition roles) {
            this.roles = roles;
            return this;
        }

        public void setCommentRequired(boolean commentRequired) {
            this.commentRequired = commentRequired;
        }

        public RoleDefinition build() {
            return new RoleDefinition(slug, values, roles, name, help, fields, commentRequired);
        }

        public Builder withField(FieldDefinition field) {
            if (this.fields == null) {
                this.fields = new ArrayList<>();
            }
            this.fields.add(field);
            return this;
        }
    }
}
