package ru.yandex.crypta.idm;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import ru.yandex.misc.lang.StringUtils;

@SuppressWarnings("ImmutableEnumChecker")
public enum Roles {

    INSTANCE();

    public static final String ADMIN = "admin";
    private final Map<String, Role> idToRole;
    private final Map<Role, String> roleToId;
    private final RoleDefinition root;

    Roles() {
        idToRole = new HashMap<>();
        roleToId = new HashMap<>();
        root = createRoles();
    }

    private RoleDefinition createLabExtRoles() {
        var labExtDirect = createRole(
                Lab.Ext.DIRECT,
                createName().en("Direct").ru("Директ"),
                createHelp().en("API for Direct").ru("API для Директа")
        );
        var labExtYaMoney = createRole(
                Lab.Ext.YAMONEY,
                createName().en("Yandex.Money").ru("Яндекс.Деньги"),
                createHelp().en("API for Yandex.Money").ru("API для Яндекс.Денег")
        );
        var labExtMarket = createRole(
                Lab.Ext.MARKET,
                createName().en("Market").ru("Маркет"),
                createHelp().en("API for Market").ru("API для Маркета")
        );
        var labExtMetrica = createRole(
                Lab.Ext.METRICA,
                createName().en("Metrica").ru("Метрика"),
                createHelp().en("API for Metrica").ru("API для Метрики")
        );
        var labExtMediaservices = createRole(
                Lab.Ext.MEDIASERVICES,
                createName().en("Mediaservices").ru("Медиасервисы"),
                createHelp().en("API for Mediaservices").ru("API для Медиасервисов")
        );
        var labExtOutstaff = createRole(
                Lab.Ext.OUTSTAFF,
                createName().en("Outstaff").ru("Аутстафф"),
                createHelp().en("API for outstaff").ru("API для аутстаффа")
        );
        return RoleDefinition.builder()
                .setName(createName().en("External API").ru("Внешние API"))
                .setHelp(createHelp().en("For integration").ru("Для интеграции"))
                .setRoles(createSubsubgroups()
                        .value(labExtDirect.getRole().getSubsubgroup(), labExtDirect.getDefinition())
                        .value(labExtMarket.getRole().getSubsubgroup(), labExtMarket.getDefinition())
                        .value(labExtMetrica.getRole().getSubsubgroup(), labExtMetrica.getDefinition())
                        .value(labExtYaMoney.getRole().getSubsubgroup(), labExtYaMoney.getDefinition())
                        .value(labExtMediaservices.getRole().getSubsubgroup(), labExtMediaservices.getDefinition())
                        .value(labExtOutstaff.getRole().getSubsubgroup(), labExtOutstaff.getDefinition()))
                .build();
    }

    private RoleDefinition createLabMatchingRoles() {
        var seeWiki = createHelp().en("(details in wiki)").ru("(подробнее в вики)");
        var nonPrivateMatching = createRole(
                Lab.Matching.NON_PRIVATE_MATCHING,
                createName()
                        .en("Matching with non-private identifiers")
                        .ru("Сопоставление с неприватными идентификаторами"),
                seeWiki
        );
        var subsubgroups = createSubsubgroups()
                .value(nonPrivateMatching.getRole().getSubsubgroup(), nonPrivateMatching.getDefinition());

        for (Lab.Matching.Mode mode : Lab.Matching.Mode.values()) {
            for (Lab.Matching.Type type : Lab.Matching.Type.values()) {
                for (Lab.Matching.Hashing hashing : type.getHashTypes()) {
                    String id = Lab.Matching.role(Lab.Matching.Privacy.PRIVATE, mode, hashing, type);
                    String enName = StringUtils
                            .capitalize(String.format("%s, %s (%s)", mode.getEn(), type.getEn(), hashing.getEn()));
                    String ruName = StringUtils
                            .capitalize(String.format("%s, %s (%s)", mode.getRu(), type.getRu(), hashing.getRu()));
                    var name = createName().en(enName).ru(ruName);
                    var role = createRole(id, RoleDefinition.builder().setName(name).setHelp(seeWiki));
                    subsubgroups.value(role.getRole().getSubsubgroup(), role.getDefinition());
                }
            }
        }


        return RoleDefinition.builder()
                .setName(createName().en("Matching").ru("Матчинг"))
                .setHelp(createHelp()
                        .en("Matching of various identifiers (cross-device, in-device, probabilistic)")
                        .ru("Матчинг идентификаторов разного типа (cross-device, in-device, probabilistic)"))
                .setRoles(subsubgroups)
                .build();
    }

    private RoleDefinition createLabRoles() {
        var labExtended = createRole(
                Lab.EXTENDED,
                createName().en("Extended access (deprecated)").ru("Расширенный доступ (deprecated)"),
                createHelp().en("Can edit segments").ru("Может редактировать сегменты")
        );
        var labAdmin = createRole(
                Lab.ADMIN,
                createName().en("Administrator").ru("Администратор"),
                createHelp()
        );
        var labAudienceSegmentsSharer = createRole(
                Lab.AUDIENCE_SEGMENTS_SHARER,
                createName().en("Share Audience exports").ru("Шеринг Аудиторных экспортов"),
                createHelp().en("Can share all Audience exports").ru("Может выдавать доступы до всех Aудиторных экспортов в Лаборатории")
        );
        var labExpressionsEditor = createRole(
                Lab.EXPRESSIONS_EDITOR,
                createName().en("Expressions editor").ru("Выражения"),
                createHelp().en("Can edit segment expressions").ru("Может редактировать выражения сегментов")
        );

        var labSegmentsRestricted = createRole(
                Lab.SEGMENTS_RESTRICTED,
                createName().en("Segments. RestrictedAccess").ru("Сегменты. Ограниченный доступ"),
                createHelp().en("Cannot create or export segments").ru("Не могут создавать или экспортировать сегменты")
        );

        return RoleDefinition.builder()
                .setName(createName().en("Laboratory").ru("Лаборатория"))
                .setHelp(createHelp().en("lab.crypta.yandex-team.ru").ru("lab.crypta.yandex-team.ru"))
                .setRoles(createSubgroups()
                        .value(labExtended.getRole().getSubgroup(), labExtended.getDefinition())  // deprecated
                        .value(labAdmin.getRole().getSubgroup(), labAdmin.getDefinition())
                        .value(labAudienceSegmentsSharer.getRole().getSubgroup(), labAudienceSegmentsSharer.getDefinition())
                        .value(labExpressionsEditor.getRole().getSubgroup(), labExpressionsEditor.getDefinition())
                        .value(labSegmentsRestricted.getRole().getSubgroup(), labSegmentsRestricted.getDefinition())
                        .value("ext", createLabExtRoles())
                        .value("matching", createLabMatchingRoles()))
                .build();
    }

    private RoleDefinition createPortalRoles() {
        var graph = createRole(Portal.GRAPH,
                createName().en("Graph").ru("Граф"),
                createHelp()
                        .en("View user identifiers graph")
                        .ru("Просмотр графа идентификаторов пользователей")
        );
        var graphPrivate = createRole(Portal.GRAPH_PRIVATE,
                createName()
                        .en("Graph with private identifiers")
                        .ru("Граф с персональными идентификаторами"),
                createHelp()
                        .en("View user identifiers graph (with private ones)")
                        .ru("Просмотр графа идентификаторов пользователей (с персональными данными)")
        );

        var graphAntifraud = createRole(Portal.GRAPH_ANTIFRAUD,
                createName()
                        .en("Graph with anti-fraud identifiers")
                        .ru("Граф с идентификаторами антифрода"),
                createHelp()
                        .en("View user identifiers graph (with private ones)")
                        .ru("Просмотр графа идентификаторов пользователей (с данными антифрода)")
        );
        var profile = createRole(Portal.PROFILE,
                createName().en("Profiles").ru("Профили"),
                createHelp()
                        .en("View user profiles")
                        .ru("Просмотр профилей пользователей")
        );
        var ltp = createRole(Portal.LONG_TERM_PROFILE,
                createName().en("Long term user profile (LTP)").ru("Долгосрочный профиль пользователя (LTP)"),
                createHelp()
                        .en("View user long term profile (LTP)")
                        .ru("Просмотр долгосрочного профиля пользователя (LTP)")
        );
        var ads = createRole(Portal.ADS,
                createName().en("Ads").ru("Реклама"),
                createHelp()
                        .en("View user ads")
                        .ru("Просмотр рекламы для пользователей")
        );
        var experiments = createRole(Portal.EXPERIMENTS,
                createName().en("Experiments").ru("Эксперименты"),
                createHelp()
                        .en("View experiments parameters")
                        .ru("Просмотр параметров экспериментов")
        );
        var tx = createRole(Portal.TX,
                createName().en("Transactions").ru("Транзакции"),
                createHelp()
                        .en("View transactional data")
                        .ru("Просмотр транзакционных данных"));

        return RoleDefinition.builder()
                .setName(createName().en("Portal").ru("Портал"))
                .setHelp(createHelp().en("crypta.yandex-team.ru").ru("crypta.yandex-team.ru"))
                .setRoles(createSubgroups()
                        .value(graph.getRole().getSubgroup(), graph.getDefinition())
                        .value(graphPrivate.getRole().getSubgroup(), graphPrivate.getDefinition())
                        .value(graphAntifraud.getRole().getSubgroup(), graphAntifraud.getDefinition())
                        .value(profile.getRole().getSubgroup(), profile.getDefinition())
                        .value(ltp.getRole().getSubgroup(), ltp.getDefinition())
                        .value(ads.getRole().getSubgroup(), ads.getDefinition())
                        .value(experiments.getRole().getSubgroup(), experiments.getDefinition())
                        .value(tx.getRole().getSubgroup(), tx.getDefinition()))
                .build();
    }

    private RoleDefinition createDataRoles() {
        var matchingBasic = createRole(
                Data.MATCHING_BASIC,
                createName().en("Technical identifiers matching").ru("Склейка технических идентификаторов"),
                createHelp().en("yandexuid, IDFA/GAID/OAID, ...").ru("yandexuid, IDFA/GAID/OAID, ...")
        );
        var profiles = createRole(
                Data.PROFILES,
                createName().en("Profiles").ru("Профили"),
                createHelp().en("Segments and socio-demographics").ru("Сегменты и соцдем")
        );

        return RoleDefinition.builder()
                .setName(createName().en("Data").ru("Данные"))
                .setHelp(createHelp().en("Access to data in all services").ru("Доступ к данным во всех сервисах "))
                .setRoles(createSubgroups()
                        .value(matchingBasic.getRole().getSubgroup(), matchingBasic.getDefinition())
                        .value(profiles.getRole().getSubgroup(), profiles.getDefinition()))
                .build();
    }

    private RoleDefinition createRoles() {
        var grab = createRole(
                Portal.GRAB,
                createName().en("Grab").ru("Grab"),
                createHelp().en("Has access to Grab").ru("Может работать с Grab")
        );
        var admin = createRole(
                ADMIN,
                createName().en("Administrator").ru("Администратор"),
                createHelp().en("Can access internal API").ru("Имеет доступ к служебным API")
        );
        var distribution = createRole(
                Portal.DISTRIBUTION,
                createName().en("Distribution").ru("Дистрибуция"),
                createHelp().en("Can access distribution components").ru("Имеет доступ к компонентам дистрибуции")
        );

        return RoleDefinition.builder()
                .group()
                .setName(createName().en("Crypta").ru("Крипта"))
                .setHelp(createHelp().en("wiki/crypta").ru("wiki/crypta"))
                .value(admin.getRole().getGroup(), admin.getDefinition())
                .value(Lab.ROOT, createLabRoles())
                .value(Portal.ROOT, createPortalRoles())
                .value(Portal.GRAB, grab.getDefinition())
                .value(Portal.DISTRIBUTION, distribution.getDefinition())
                .value(Data.ROOT, createDataRoles())
                .build();
    }

    public RoleDefinition root() {
        return root;
    }

    public boolean hasRole(Role role) {
        return idToRole.containsValue(role);
    }

    public boolean hasId(String id) {
        return idToRole.containsKey(id);
    }

    public String toId(Role role) {
        return roleToId.get(role);
    }

    public Role fromId(String id) {
        return idToRole.get(id);
    }

    private CreatedRole createRole(String id, RoleDefinition.Builder definition) {
        Role role = Role.fromUnderscoreString(id).build();
        idToRole.put(id, role);
        roleToId.put(role, id);
        return new CreatedRole(role, definition.build());
    }

    private CreatedRole createRole(String id, LocalizedString.Builder name, LocalizedString.Builder help) {
        return createRole(id, RoleDefinition.builder().setName(name).setHelp(help));
    }

    private LocalizedString.Builder createName() {
        return LocalizedString.builder();
    }

    private LocalizedString.Builder createHelp() {
        return LocalizedString.builder();
    }

    private RoleDefinition.Builder createSubgroups() {
        return RoleDefinition.builder().subgroup();
    }

    private RoleDefinition.Builder createSubsubgroups() {
        return RoleDefinition.builder().subsubgroup();
    }

    private static class CreatedRole {

        private final Role role;
        private final RoleDefinition definition;

        private CreatedRole(Role role, RoleDefinition definition) {
            this.role = role;
            this.definition = definition;
        }

        public Role getRole() {
            return role;
        }

        public RoleDefinition getDefinition() {
            return definition;
        }
    }

    public static final class Lab {

        public static final String EXTENDED = "lab_extended";  // deprecated

        public static final String ADMIN = "lab_admin";
        public static final String AUDIENCE_SEGMENTS_SHARER = "lab_audience-segments-sharer";
        public static final String EXPRESSIONS_EDITOR = "lab_expressions-editor";
        public static final String SEGMENTS_RESTRICTED = "lab_segments-restricted";
        public static final String SEGMENTS_NOT_ALLOWED_META_ROLE = "lab_segments-not-allowed";
        private static final String ROOT = "lab";

        public static final class Ext {
            public static final String MARKET = "lab_ext_market";
            public static final String DIRECT = "lab_ext_direct";
            public static final String YAMONEY = "lab_ext_yamoney";
            public static final String METRICA = "lab_ext_metrica";
            public static final String MEDIASERVICES = "lab_ext_mediaservices";
            public static final String OUTSTAFF = "lab_ext_outstaff";
        }

        public static final class Matching {
            public static final String NON_PRIVATE_MATCHING = "lab_matching_non-private/matching";

            public static String role(Privacy privacy, Mode mode, Hashing hashing, Type type) {
                return String.format("lab_matching_%s/%s/%s/%s", privacy.getId(), mode.getId(), type.getId(),
                        hashing.getId());
            }

            public enum Privacy {
                PRIVATE("private", "private", "персональные"),
                NON_PRIVATE("non-private", "non-private", "неперсональные");

                private final String id;
                private final String en;
                private final String ru;

                Privacy(String id, String en, String ru) {
                    this.id = id;
                    this.en = en;
                    this.ru = ru;
                }

                public String getId() {
                    return id;
                }

                public String getEn() {
                    return en;
                }

                public String getRu() {
                    return ru;
                }
            }

            public enum Mode {
                CONVERTING_GROUP("converting", "сonverting", "конвертация"),
                MATCHING_SIDE_BY_SIDE("matching", "matching", "сопоставление");

                private final String id;
                private final String en;
                private final String ru;

                Mode(String id, String en, String ru) {
                    this.id = id;
                    this.en = en;
                    this.ru = ru;
                }

                public String getId() {
                    return id;
                }

                public String getEn() {
                    return en;
                }

                public String getRu() {
                    return ru;
                }
            }

            public enum Hashing {
                HASHED("hashed", "hashed", "с хэшированием"),
                NON_HASHED("non-hashed", "non-hashed", "без хэширования");

                private final String id;
                private final String en;
                private final String ru;

                Hashing(String id, String en, String ru) {
                    this.id = id;
                    this.en = en;
                    this.ru = ru;
                }

                public String getId() {
                    return id;
                }

                public String getEn() {
                    return en;
                }

                public String getRu() {
                    return ru;
                }
            }

            public enum Type {
                PUID("puid", "Passport ID (puid)", "ID Паспорта (puid)", List.of(Hashing.NON_HASHED)),
                LOGIN("login", "login", "логин"),
                EMAIL("email", "email", "email"),
                PHONE("phone", "phone", "телефон");

                private final String id;
                private final String en;
                private final String ru;
                private List<Hashing> hashTypes = Arrays.asList(Hashing.values());

                Type(String id, String en, String ru) {
                    this.id = id;
                    this.en = en;
                    this.ru = ru;
                }

                Type(String id, String en, String ru, List<Hashing> hashTypes) {
                    this.id = id;
                    this.en = en;
                    this.ru = ru;
                    this.hashTypes = hashTypes;
                }

                public String getId() {
                    return id;
                }

                public String getEn() {
                    return en;
                }

                public String getRu() {
                    return ru;
                }

                public List<Hashing> getHashTypes() {
                    return hashTypes;
                }
            }

        }
    }

    public static final class Portal {

        public static final String GRAB = "grab";
        public static final String DISTRIBUTION = "distribution";
        public static final String GRAPH = "portal_graph";
        public static final String GRAPH_PRIVATE = "portal_graph-private";
        public static final String GRAPH_ANTIFRAUD = "portal_graph-antifraud";
        public static final String PROFILE = "portal_profile";
        public static final String LONG_TERM_PROFILE = "portal_ltp";
        public static final String ADS = "portal_ads";
        public static final String TX = "portal_tx";
        public static final String EXPERIMENTS = "portal_experiments";
        public static final String ROOT = "portal";

    }

    public static final class Data {

        public static final String ROOT = "data";

        public static final String MATCHING_BASIC = "data_matching-basic";
        public static final String PROFILES = "data_profiles";

    }

}
