package ru.yandex.crypta.idm.rest.resource.idm;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.inject.Inject;
import javax.validation.constraints.NotNull;
import javax.ws.rs.FormParam;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.SecurityContext;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

import ru.yandex.crypta.clients.idm.RoleService;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.common.ws.jersey.JsonUtf8;
import ru.yandex.crypta.idm.Role;
import ru.yandex.crypta.idm.Roles;
import ru.yandex.crypta.idm.rest.resource.idm.parameter.LoginParameter;
import ru.yandex.crypta.idm.rest.resource.idm.parameter.RoleParameter;
import ru.yandex.crypta.idm.rest.resource.idm.response.LoginRolesResponse;
import ru.yandex.crypta.idm.rest.resource.idm.response.LoginToRoleResponse;
import ru.yandex.crypta.idm.rest.resource.idm.response.RolesResponse;
import ru.yandex.crypta.idm.rest.resource.idm.response.SimpleResponse;
import ru.yandex.crypta.lib.proto.TDevelopmentFlags;
import ru.yandex.crypta.lib.proto.TIdmTvmConfig;

@Path("idm")
@Api(tags = {"idm"})
@Produces(JsonUtf8.MEDIA_TYPE)
public class IdmResource {

    private final RoleService roleService;
    private final TIdmTvmConfig idmTvmConfig;
    private final boolean checkServiceTicket;

    @Context
    private SecurityContext securityContext;

    @Inject
    public IdmResource(TDevelopmentFlags flags, RoleService roleService, TIdmTvmConfig idmTvmConfig) {
        this.roleService = roleService;
        this.checkServiceTicket = !flags.getDisableTvm();
        this.idmTvmConfig = idmTvmConfig;
    }

    @POST
    @ApiOperation(value = "Removes role")
    @Path("remove-role")
    public SimpleResponse removeRole(
            @NotNull @FormParam("login") @ApiParam(value = "User login") LoginParameter login,
            @NotNull @FormParam("role") @ApiParam(value = "Role in json") RoleParameter role
    )
    {
        verifyIdmAuth();
        roleService.removeRole(login.getLogin(), role.getRole());
        return SimpleResponse.ok();
    }

    @POST
    @ApiOperation(value = "Adds new role")
    @Path("add-role")
    public SimpleResponse addRole(
            @NotNull @FormParam("login") @ApiParam(value = "User login") LoginParameter login,
            @NotNull @FormParam("role") @ApiParam(value = "Role in json") RoleParameter role
    )
    {
        verifyIdmAuth();
        roleService.addRole(login.getLogin(), role.getRole());
        return SimpleResponse.ok();
    }

    @GET
    @ApiOperation(value = "Retrieves all available roles", nickname = "idm-info")
    @Path("info")
    public RolesResponse info()
    {
        verifyIdmAuth();
        return new RolesResponse(Roles.INSTANCE.root());
    }

    @GET
    @ApiOperation(value = "Retrieves roles per login")
    @Path("get-all-roles")
    public LoginToRoleResponse getAllRoles()
    {
        verifyIdmAuth();
        List<LoginToRoleResponse.LoginWithRoles> loginsWithRoles = roleService
                .getAllLoginRoles()
                .entrySet()
                .stream().map(entry -> {
                    String login = entry.getKey();
                    List<Role> roles = new ArrayList<>(entry.getValue());
                    return new LoginToRoleResponse.LoginWithRoles(login, roles);
                }).collect(Collectors.toList());
        return new LoginToRoleResponse(0, loginsWithRoles);
    }

    @GET
    @ApiOperation(value = "Retrieve roles per login")
    @Path("get-roles")
    public LoginToRoleResponse getRolesFromHeaders()
    {
        return getAllRoles();
    }

    private Optional<Role> exclusivelyRestrictedSegmentsRole(List<Role> baseRoles) {
        Role restrictedRole = Role.fromUnderscoreString(Roles.Lab.SEGMENTS_RESTRICTED).build();
        Role audienceSharerRole = Role.fromUnderscoreString(Roles.Lab.AUDIENCE_SEGMENTS_SHARER).build();
        Role expressionsRole = Role.fromUnderscoreString(Roles.Lab.EXPRESSIONS_EDITOR).build();

        if (baseRoles.contains(restrictedRole) && !baseRoles.contains(audienceSharerRole) && !baseRoles.contains(expressionsRole)) {
            return Optional.of(Role.fromUnderscoreString(Roles.Lab.SEGMENTS_NOT_ALLOWED_META_ROLE).build());
        }

        return Optional.empty();
    }

    @GET
    @ApiOperation(value = "Retrieve roles of given login")
    @Path("roles/{login}")
    public LoginRolesResponse getRoles(
            @PathParam("login") @NotNull LoginParameter login
    )
    {
        List<Role> roles = roleService.getRoles(login.getLogin());
        return new LoginRolesResponse(login.getLogin(), roles);
    }

    @GET
    @ApiOperation(value = "Retrieve roles of authenticated user")
    @Path("my-roles")
    public LoginRolesResponse getMyRoles() {
        String login = securityContext.getUserPrincipal().getName();
        List<Role> roles = roleService.getRoles(login);

        // Meta role for CRYPTA-15435
        var restrictedMetaRole = exclusivelyRestrictedSegmentsRole(roles);
        restrictedMetaRole.ifPresent(roles::add);

        return new LoginRolesResponse(login, roles);
    }

    private void verifyIdmAuth() {
        if (!checkServiceTicket) {
            return;
        }

        if (!securityContext.getUserPrincipal().getName().equals(String.valueOf(idmTvmConfig.getSourceTvmId()))) {
            throw Exceptions.notAuthenticated("Wrong TVM ticket");
        }
    }

}
