package ru.yandex.crypta.idm.rest.resource.other;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.function.Consumer;

import javax.annotation.security.RolesAllowed;
import javax.inject.Inject;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Response;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import io.swagger.annotations.Api;
import org.flywaydb.core.Flyway;
import org.flywaydb.core.api.MigrationInfo;
import org.flywaydb.core.api.MigrationInfoService;
import org.flywaydb.core.api.callback.BaseCallback;
import org.flywaydb.core.api.callback.Callback;
import org.flywaydb.core.api.callback.Context;
import org.slf4j.helpers.MessageFormatter;

import ru.yandex.crypta.common.ws.jersey.JsonUtf8;
import ru.yandex.crypta.idm.Roles;
import ru.yandex.crypta.idm.migrations.Migrations;

@Path("flyway")
@Api(tags = {"flyway"})
@Produces(JsonUtf8.MEDIA_TYPE)
@RolesAllowed(Roles.ADMIN)
public class FlywayResource {

    private final Flyway flyway;

    public static class Event {
        private final String message;
        private final Instant time;

        public Event(String message) {
            this.message = message;
            this.time = Instant.now();
        }

        public String getMessage() {
            return message;
        }

        public Long getTime() {
            return time.toEpochMilli();
        }
    }

    public static class History {

        private List<Event> events = new ArrayList<>();

        public List<Event> getEvents() {
            return events;
        }

        @JsonIgnore
        public Callback getFlywayCallback() {
            return new BaseCallback() {
                @Override
                public void handle(org.flywaydb.core.api.callback.Event event, Context context) {
                    var info = context.getMigrationInfo();
                    switch (event) {
                        case BEFORE_CLEAN:
                            add("Going to clean up");
                            break;
                        case AFTER_CLEAN:
                            add("Cleaning done");
                            break;
                        case BEFORE_MIGRATE:
                            add("Going to migrate");
                            break;
                        case AFTER_MIGRATE:
                            add("Migration done");
                            break;
                        case BEFORE_EACH_MIGRATE:
                            add("Applying {} migration {} (checksum {})",
                                    info.getType().name(), info.getScript(), info.getChecksum());
                            break;
                        case AFTER_EACH_MIGRATE:
                            add("Applied {} migration {} (checksum {})",
                                    info.getType().name(), info.getScript(), info.getChecksum());
                            break;
                        case BEFORE_VALIDATE:
                            add("Validating");
                            break;
                        case AFTER_VALIDATE:
                            add("Validated");
                            break;
                        case BEFORE_REPAIR:
                            add("Going to repair migrations");
                            break;
                        case AFTER_REPAIR:
                            add("Repaired migrations");
                            break;
                        case BEFORE_BASELINE:
                            add("Marking current version as a baseline");
                            break;
                        case AFTER_BASELINE:
                            add("Marked current version as a baseline");
                            break;
                        default:
                            add(event.getId());
                    }
                }
            };
        }

        public void addException(Exception exception) {
            add(exception.getMessage());
        }

        private void add(String message, Object... params) {
            events.add(new Event(format(message, params)));
        }

        private String format(String format, Object... params) {
            return MessageFormatter.arrayFormat(format, params).getMessage();
        }

    }

    @Inject
    public FlywayResource(Migrations migrations) {
        this.flyway = migrations.getFlyway();
    }

    @POST
    @Path("baseline")
    public Response baseline() {
        return withHistory(Flyway::baseline);
    }

    @POST
    @Path("migrate")
    public Response migrate() {
        return withHistory(Flyway::migrate);
    }

    private Response withHistory(Consumer<Flyway> flywayAction) {
        History history = new History();

        var configuredFlyway = Flyway.configure()
                .configuration(flyway.getConfiguration())
                .callbacks(history.getFlywayCallback())
                .load();

        try {
            flywayAction.accept(configuredFlyway);
        } catch (Exception ex) {
            history.addException(ex);
            return Response.serverError().entity(history).build();
        }
        return Response.ok().entity(history).build();
    }

    public static class FlywayInfo {
        private final MigrationInfoService info;

        public FlywayInfo(MigrationInfoService info) {
            this.info = info;
        }

        @JsonSerialize(typing = JsonSerialize.Typing.STATIC)
        public List<MigrationInfo> getApplied() {
            return Arrays.asList(info.applied());
        }

        @JsonSerialize(typing = JsonSerialize.Typing.STATIC)
        public List<MigrationInfo> getPending() {
            return Arrays.asList(info.pending());
        }

    }

    @GET
    @Path("info")
    public FlywayInfo info() {
        return new FlywayInfo(flyway.info());
    }

}
