#include "graph.h"

#include "id_serializer.h"
#include "id_utils.h"

using namespace NCrypta::NIS;

TGraph::~TGraph() {
    Clear();
}

TNode* TGraph::CreateNode(const TId& id) {
    return CreateNode(id.Type, id.Value);
}

TNode* TGraph::CreateNode(const TString& type, const TString& value) {
    TId id(type, value);

    Y_ENSURE(IsUnknownId(id) || !IdNodes.contains(id), "Node " << TIdSerializer::Serialize(id) << " is already in graph");

    auto nodeHolder = MakeHolder<TNode>(std::move(id));
    auto* node = nodeHolder.Get();

    Nodes.push_back(node);
    Y_UNUSED(nodeHolder.Release());

    NodeIndices[node] = Nodes.size() - 1;

    if (!IsUnknownId(node->Id)) {
        IdNodes[node->Id] = node;
    }

    return node;
}

TEdge* TGraph::CreateEdge(size_t nodeIndex1, size_t nodeIndex2) {
    Y_ENSURE(nodeIndex1 < Nodes.size());
    Y_ENSURE(nodeIndex2 < Nodes.size());

    return CreateEdge(GetNode(nodeIndex1), GetNode(nodeIndex2));
}

TEdge* TGraph::CreateEdge(TNode* node1, TNode* node2) {
    Y_ENSURE(node1 != nullptr);
    Y_ENSURE(node2 != nullptr);
    Y_ENSURE(node1 != node2, "Loop edge: " << TIdSerializer::Serialize(node1->Id) << " to itself");

    auto edgeHolder = MakeHolder<TEdge>(node1, node2);
    auto* edge = edgeHolder.Get();

    Edges.push_back(edge);
    Y_UNUSED(edgeHolder.Release());

    node1->Edges.push_back(edge);
    node2->Edges.push_back(edge);

    return edge;
}

TNode* TGraph::GetNode(size_t nodeIndex) const {
    Y_ENSURE(nodeIndex < Nodes.size());

    return Nodes[nodeIndex];
}

size_t TGraph::IndexOf(const TNode* node) const {
    Y_ENSURE(node != nullptr);

    return NodeIndices.at(node);
}

TNode* TGraph::FindNode(const TId& id) const {
    Y_ENSURE(!IsUnknownId(id), "Can not search for unknown id");

    auto it = IdNodes.find(id);
    return it == IdNodes.end() ? nullptr : it->second;
}

bool TGraph::HasNode(const TId& id) const {
    return FindNode(id) != nullptr;
}

bool TGraph::IsEmpty() const {
    return Nodes.empty();
}

const TGraph::TNodes& TGraph::GetNodes() const {
    return Nodes;
}

const TGraph::TEdges& TGraph::GetEdges() const {
    return Edges;
}

const TAttributes& TGraph::GetAttributes() const {
    return Attributes;
}

TAttributes& TGraph::GetAttributes() {
    return Attributes;
}

bool TGraph::AreAdjacent(size_t nodeIndex1, size_t nodeIndex2) const {
    Y_ENSURE(nodeIndex1 < Nodes.size());
    Y_ENSURE(nodeIndex2 < Nodes.size());

    return AreAdjacent(GetNode(nodeIndex1), GetNode(nodeIndex2));
}

bool TGraph::AreAdjacent(const TNode* node1, const TNode* node2) const {
    Y_ENSURE(node1 != nullptr);
    Y_ENSURE(node2 != nullptr);

    const auto& edges1 = node1->Edges;
    auto it = std::find_if(edges1.begin(), edges1.end(), [&](const auto& e) { return e->GetAdjacentNode(node1) == node2; });

    return it != edges1.end();
}

TGraph::TCryptaId TGraph::GetId() const {
    return Id;
}

void TGraph::SetId(TCryptaId id) {
    Id = id;
}

void TGraph::Merge(TGraph&& other) {
    Y_ENSURE(this != &other, "Can not merge graph with itself");

    size_t originalSize = Nodes.size();

    Nodes.reserve(Nodes.size() + other.Nodes.size());
    NodeIndices.reserve(NodeIndices.size() + other.NodeIndices.size());
    IdNodes.reserve(IdNodes.size() + other.IdNodes.size());
    Edges.reserve(Edges.size() + other.Edges.size());

    Nodes.insert(Nodes.end(), std::make_move_iterator(other.Nodes.begin()), std::make_move_iterator(other.Nodes.end()));
    std::fill(other.Nodes.begin(), other.Nodes.end(), nullptr);

    for (auto& kvp : other.NodeIndices) {
        kvp.second += originalSize;
        NodeIndices.insert(kvp);
    }

    for (const auto& otherNode: other.IdNodes) {
        Y_ENSURE(IdNodes.insert(otherNode).second, "Merging intersecting graphs");
    }

    Edges.insert(Edges.end(), std::make_move_iterator(other.Edges.begin()), std::make_move_iterator(other.Edges.end()));
    std::fill(other.Edges.begin(), other.Edges.end(), nullptr);

    other.Clear();
}

void TGraph::Clear() {
    for (auto* node : Nodes) {
        delete node;
    }

    for (auto* edge : Edges) {
        delete edge;
    }

    Id = 0;
    Nodes.clear();
    NodeIndices.clear();
    IdNodes.clear();
    Edges.clear();
}

size_t TGraph::MemoryUsage() const {
    // TODO handle case when one TString is shared between several attributes
    size_t result = sizeof(TGraph);
    auto sizeOfAttributes = [](const TAttributes& attrs) -> size_t {
        size_t r = 0;
        for (auto& attr : attrs) {
            r += attr.first.capacity();
            r += attr.second.capacity();
        }
        r += sizeof(TAttributes::value_type) * attrs.size();
        return r;
    };

    for (auto* node : Nodes) {
        result += sizeof(*node);
        result += node->Id.Type.capacity();
        result += node->Id.Value.capacity();
        result += sizeOfAttributes(node->Attributes);
        result += node->Edges.capacity() * sizeof(TEdge*);
    }
    result += Nodes.capacity() * sizeof(TNode*);

    for (auto* edge : Edges) {
        result += sizeof(*edge);
        result += sizeOfAttributes(edge->Attributes);
    }
    result += sizeOfAttributes(Attributes);
    result += sizeof(decltype(NodeIndices)::value_type) * NodeIndices.size();
    result += sizeof(decltype(IdNodes)::value_type) * IdNodes.size();
    return result;
}
