#include "graph_comparator.h"

#include <algorithm>

using namespace NCrypta::NIS;

bool TGraphComparator::Equal(const TGraph& graph1, const TGraph& graph2) {
    if (!BasicEqual(graph1, graph2)) {
        return false;
    }

    for (const auto& node1 : graph1.GetNodes()) {
        const auto* node2 = graph2.FindNode(node1->Id);

        if (node2 == nullptr) {
            return false;
        }

        if (node1->Attributes != node2->Attributes) {
            return false;
        }

        for (const auto& edge1 : node1->Edges) {
            if (node1 != edge1->Node1) {
                continue;
            }

            const auto& edges2 = node2->Edges;

            auto it = std::find_if(edges2.begin(), edges2.end(), [&](const auto& edge2) {
                return (edge1->GetAdjacentNode(node1)->Id == edge2->GetAdjacentNode(node2)->Id) &&
                       (edge1->Attributes == edge2->Attributes);
            });

            if (it == edges2.end()) {
                return false;
            }
        }
    }

    return true;
}

bool TGraphComparator::OrderedEqual(const TGraph& graph1, const TGraph& graph2) {
    if (!BasicEqual(graph1, graph2)) {
        return false;
    }

    const auto& nodes1 = graph1.GetNodes();
    const auto& nodes2 = graph2.GetNodes();

    for (size_t i = 0; i < nodes1.size(); i++) {
        const auto* node1 = nodes1[i];
        const auto* node2 = nodes2[i];

        if ((node1->Id != node2->Id) || (node1->Attributes != node2->Attributes)) {
            return false;
        }
    }

    const auto& edges1 = graph1.GetEdges();
    const auto& edges2 = graph2.GetEdges();

    for (size_t i = 0; i < edges1.size(); i++) {
        const auto* edge1 = edges1[i];
        const auto* edge2 = edges2[i];

        if ((graph1.IndexOf(edge1->Node1) != graph2.IndexOf(edge2->Node1)) ||
            (graph1.IndexOf(edge1->Node2) != graph2.IndexOf(edge2->Node2)) ||
            (edge1->Attributes != edge2->Attributes))
        {
            return false;
        }
    }

    return true;
}

bool TGraphComparator::BasicEqual(const TGraph& graph1, const TGraph& graph2) {
    return (graph1.GetId() == graph2.GetId()) &&
           (graph1.GetAttributes() == graph2.GetAttributes()) &&
           (graph1.GetNodes().size() == graph2.GetNodes().size()) &&
           (graph1.GetEdges().size() == graph2.GetEdges().size());
}
