package ru.yandex.crypta.lab;

import java.util.Objects;
import java.util.stream.Stream;

import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.lab.proto.EHashingMethod;
import ru.yandex.crypta.lab.proto.ELabIdentifierType;
import ru.yandex.crypta.lab.utils.SchemaField;
import ru.yandex.crypta.lib.proto.identifiers.EIdType;

public enum Identifier {

    YANDEXUID("yandexuid", ELabIdentifierType.LAB_ID_YANDEXUID, EHashingMethod.HM_IDENTITY,
            EIdType.YANDEXUID),
    IDFA_GAID("idfa_gaid", ELabIdentifierType.LAB_ID_IDFA_GAID, EHashingMethod.HM_IDENTITY,
            EIdType.IDFA_GAID),
    EMAIL("email", ELabIdentifierType.LAB_ID_EMAIL, EHashingMethod.HM_IDENTITY, EIdType.EMAIL),
    PHONE("phone", ELabIdentifierType.LAB_ID_PHONE, EHashingMethod.HM_IDENTITY, EIdType.PHONE),
    EMAIL_MD5("email_md5", ELabIdentifierType.LAB_ID_EMAIL, EHashingMethod.HM_MD5, EIdType.EMAIL_MD5),
    PHONE_MD5("phone_md5", ELabIdentifierType.LAB_ID_PHONE, EHashingMethod.HM_MD5, EIdType.PHONE_MD5),
    LOGIN("login", ELabIdentifierType.LAB_ID_LOGIN, EHashingMethod.HM_IDENTITY, EIdType.LOGIN),
    MM_DEVICE_ID("mm_device_id", ELabIdentifierType.LAB_ID_MM_DEVICE_ID, EHashingMethod.HM_IDENTITY,
            EIdType.MM_DEVICE_ID),
    PUID("puid", ELabIdentifierType.LAB_ID_PUID, EHashingMethod.HM_IDENTITY, EIdType.PUID),
    UUID("uuid", ELabIdentifierType.LAB_ID_UUID, EHashingMethod.HM_IDENTITY, EIdType.UUID),
    CRYPTAID("crypta_id", ELabIdentifierType.LAB_ID_CRYPTA_ID, EHashingMethod.HM_IDENTITY, EIdType.CRYPTA_ID),
    DIRECT_CLIENT_ID("client_id", ELabIdentifierType.LAB_ID_DIRECT_CLIENT_ID, EHashingMethod.HM_IDENTITY,
            EIdType.DIRECT_CLIENT_ID);

    private final String name;
    private final ELabIdentifierType labType;
    private final EHashingMethod hashingMethod;
    private final EIdType idType;

    Identifier(String name, ELabIdentifierType labType, EHashingMethod hashingMethod, EIdType idType) {
        this.name = name;
        this.labType = labType;
        this.hashingMethod = hashingMethod;
        this.idType = idType;
    }

    public static Identifier byName(String name) {
        return Stream.of(values()).filter(x -> x.getName().equals(name))
                .findFirst()
                .orElseThrow(() -> Exceptions.illegal("No such identifier " + name));
    }

    public static Identifier byLabType(ELabIdentifierType type) {
        return Stream.of(values()).filter(x -> x.getLabType().equals(type))
                .findFirst()
                .orElseThrow(() -> Exceptions
                        .wrongRequestException("Unsupported identifier " + type.name(), "UNSUPPORTED_IDENTIFIER"));
    }

    public static Identifier byIdType(EIdType type) {
        return Stream.of(values()).filter(x -> x.getIdType().equals(type))
                .findFirst()
                .orElseThrow(() -> Exceptions
                        .wrongRequestException("Unsupported identifier " + type.name(), "UNSUPPORTED_IDENTIFIER"));
    }

    public boolean matches(SchemaField schemaField) {
        return Objects.equals(name, schemaField.getName());
    }

    public String getName() {
        return name;
    }

    public ELabIdentifierType getLabType() {
        return labType;
    }

    public EIdType getIdType() {
        return idType;
    }

    public EHashingMethod getHashingMethod() {
        return hashingMethod;
    }
}
