package ru.yandex.crypta.lab.base;

import java.util.Map;
import java.util.function.Function;

import javax.ws.rs.core.SecurityContext;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.crypta.clients.pgaas.PostgresClient;
import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.common.data.EnhancedDSLContext;
import ru.yandex.crypta.common.ws.auth.EmptySecurityContext;
import ru.yandex.crypta.lab.WithOptions;
import ru.yandex.crypta.lab.tables.Tables;
import ru.yandex.crypta.lib.proto.EEnvironment;

public abstract class BaseService<T> implements WithOptions<T> {

    private static final Map<EEnvironment, String> ENVIRONMENT_TO_SANDBOX = Cf.map(
            EEnvironment.ENV_DEVELOP, "develop",
            EEnvironment.ENV_TESTING, "testing",
            EEnvironment.ENV_PRODUCTION, "stable"
    );

    private final EEnvironment environment;
    private final PostgresClient sql;
    private Language language = Language.en();
    private SecurityContext securityContext = new EmptySecurityContext();

    protected BaseService(EEnvironment environment, PostgresClient sql) {
        this.environment = environment;
        this.sql = sql;
    }

    @Override
    @SuppressWarnings("unchecked")
    public T withLanguage(Language language) {
        BaseService cloned = (BaseService) clone();
        cloned.securityContext = securityContext;
        cloned.language = language;
        return (T) cloned;
    }

    @Override
    @SuppressWarnings("unchecked")
    public T withSecurityContext(SecurityContext securityContext) {
        BaseService cloned = (BaseService) clone();
        cloned.securityContext = securityContext;
        cloned.language = language;
        return (T) cloned;
    }

    protected Language language() {
        return language;
    }

    protected SecurityContext securityContext() {
        return securityContext;
    }

    protected PostgresClient sql() {
        return sql;
    }

    protected EnhancedDSLContext sqlDsl() {
        return new EnhancedDSLContext(sql().getJooqConfiguration());
    }

    protected Tables tables() {
        return new Tables(sql.getJooqConfiguration(), securityContext());
    }

    protected EEnvironment environment() {
        return environment;
    }

    protected String getSandboxEnvironment() {
        return ENVIRONMENT_TO_SANDBOX.get(environment());
    }

    @Override
    public abstract T clone();

    protected <R> R withSqlTransaction(Function<Tables, R> body) {
        EnhancedDSLContext.TransactionBody<R> transactionBody =
                (sqlTransaction) -> body.apply(new Tables(sqlTransaction, securityContext()));
        return sqlDsl().withTransaction(transactionBody);
    }

    protected <R> R withLongSqlTransaction(Function<Tables, R> body) {
        // CRYPTAUP-1711 deleting outdated samples takes time
        EnhancedDSLContext.TransactionBody<R> transactionBody = (sqlTransaction) -> {
            sqlTransaction.settings().setQueryTimeout(3600);
            return body.apply(new Tables(sqlTransaction, securityContext()));
        };
        return sqlDsl().withTransaction(transactionBody);
    }

}
