package ru.yandex.crypta.lab.base;

import java.io.InputStream;
import java.time.Duration;
import java.util.Optional;
import java.util.function.Function;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.crypta.clients.pgaas.PostgresClient;
import ru.yandex.crypta.lab.utils.YtParameters;
import ru.yandex.crypta.lib.proto.EEnvironment;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.cypress.Cypress;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.transactions.utils.YtTransactionsUtils;
import ru.yandex.inside.yt.kosher.transactions.Transaction;
import ru.yandex.inside.yt.kosher.ytree.YTreeListNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

public abstract class BaseYtService<T> extends BaseService<T> {
    public static final String ROW_COUNT = "row_count";
    public static final String SCHEMA = "schema";

    private final YtService yt;

    protected BaseYtService(EEnvironment environment, PostgresClient sql, YtService yt) {
        super(environment, sql);
        this.yt = yt;
    }

    protected <R> R withYtTransaction(Function<Transaction, R> body) {
        try {
            return YtTransactionsUtils.withTransaction(yt.getHahn(), Duration.ofSeconds(10), body);
        } catch (RuntimeException e) {
            throw e;
        }
    }

    protected <R> R withLongYtTransaction(Function<Transaction, R> body) {
        // CRYPTAUP-1711 deleting outdated samples takes time
        try {
            return YtTransactionsUtils.withTransaction(yt.getHahn(), Duration.ofHours(1), body);
        } catch (RuntimeException e) {
            throw e;
        }
    }

    protected Yt yt() {
        return yt.getHahn();
    }

    protected YtService ytService() {
        return yt;
    }

    protected Optional<GUID> optionalId(Transaction ytTransaction) {
        return Optional.of(ytTransaction.getId());
    }

    protected Cypress cypress() {
        return yt().cypress();
    }

    protected void createDirectory(Transaction ytTransaction, YPath path, YTreeListNode acl) {
        cypress().create(optionalId(ytTransaction), YtParameters.DO_NOT_PING, path,
                CypressNodeType.MAP, YtParameters.RECURSIVE, YtParameters.IGNORE_EXISTING, Cf.map("acl", acl));
    }

    protected GUID copyTable(Transaction ytTransaction, YPath source, YPath destination) {
        return cypress().copy(optionalId(ytTransaction).orElse(null), YtParameters.DO_NOT_PING, source, destination,
                YtParameters.RECURSIVE, YtParameters.FORCE, YtParameters.DO_NOT_PRESERVE_ACCOUNT);
    }

    protected YTreeNode adoptableTable(YPath path, Transaction ytTransaction) {
        return cypress().get(Optional.of(ytTransaction.getId()), YtParameters.DO_NOT_PING, path, Cf.set(ROW_COUNT, SCHEMA));
    }

    protected void deleteDirectory(Transaction ytTransaction, YPath path) {
        if (cypress().exists(optionalId(ytTransaction), YtParameters.DO_NOT_PING, path)) {
            cypress().remove(optionalId(ytTransaction), YtParameters.DO_NOT_PING, path, YtParameters.RECURSIVE, YtParameters.FORCE);
        }
    }

    protected void deleteTable(Transaction ytTransaction, YPath path) {
        cypress().remove(optionalId(ytTransaction), YtParameters.DO_NOT_PING, path);
    }

    protected void deleteTableIfExists(Transaction ytTransaction, String path) {
        Optional<YPath> optionalYPath = safePath(path);
        if (optionalYPath.isPresent() && existsTable(ytTransaction, optionalYPath.get())) {
            deleteTable(ytTransaction, optionalYPath.get());
        }
    }

    protected Optional<YPath> safePath(String path) {
        try {
            return Optional.of(YPath.simple(path));
        } catch (IllegalArgumentException e) {
            return Optional.empty();
        }
    }

    protected boolean existsTable(Transaction ytTransaction, YPath yPath) {
        return cypress().exists(optionalId(ytTransaction), YtParameters.DO_NOT_PING, yPath);
    }

    protected void writeFile(Transaction ytTransaction, YPath destinationFile, InputStream stream) {
        yt().files().write(Optional.of(ytTransaction.getId()), true, destinationFile, stream);
    }
}
