package ru.yandex.crypta.lab.formatters;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import org.jetbrains.annotations.NotNull;

import ru.yandex.crypta.lib.lemmer.Lemmer;
import ru.yandex.crypta.lib.word.rule.WordRuleParser;

public class WordRuleFormatter extends BooleanRuleFormatter{
    final public static String AMBIGUOUS_TAG = "word.ambiguous";

    public static final WordRuleFormatter instance = new WordRuleFormatter();

    @Override
    protected BooleanRuleFormatter.ExpressionNode visitLemmaExpr(@NotNull WordRuleParser.LemmaExprContext ctx) {
        var lemma = ctx.getText().toLowerCase();
        var normalizedLemmas = Lemmer.getLemmas(lemma, "ru");
        if (normalizedLemmas.contains(lemma)) {
            return new ExpressionNode(lemma);
        } else if (normalizedLemmas.size() == 1) {
            return new ExpressionNode(normalizedLemmas.iterator().next());
        } else {
            return new ExpressionNode(
                OR,
                normalizedLemmas.stream().sorted().map(ExpressionNode::new).collect(Collectors.toList()),
                List.of(AMBIGUOUS_TAG)
            );
        }
    }

    private String addAndOperator(String value) {
        // remove spaces around operators
        value = value.replace(" OR ", "OR");
        value = value.replace(" AND ", "AND");
        value = value.replace("NOT ", "NOT");

        // replace left spaces with AND
        value = value.replace(" ", "AND");

        // restore spaces around operators
        value = value.replace("OR", " OR ");
        value = value.replace("AND", " AND ");
        value = value.replace("NOT", "NOT ");

        return value;
    }

    @Override
    public FormattedLine format(final String wordRule) {
        try {
            var normalizedRule = getNormalizedRule(addAndOperator(wordRule));
            return FormattedLine.line(wordRule, normalizedRule.toString(), new ArrayList<>(normalizedRule.getTags()));
        } catch (ParsingException e) {
            return FormattedLine.error(wordRule, e.getMessage());
        }
    }
}
