package ru.yandex.crypta.lab.job;

import java.time.Duration;
import java.util.Collection;
import java.util.function.Consumer;

import javax.inject.Inject;

import com.google.common.collect.Lists;
import com.google.protobuf.Message;
import org.quartz.DisallowConcurrentExecution;
import org.quartz.Job;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.crypta.clients.pgaas.PostgresClient;
import ru.yandex.crypta.common.data.GenericTable;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.common.ws.auth.ServiceSecurityContext;
import ru.yandex.crypta.common.ws.solomon.Solomon;
import ru.yandex.crypta.lab.tables.Tables;
import ru.yandex.crypta.lab.utils.Paths;
import ru.yandex.crypta.lib.proto.EEnvironment;
import ru.yandex.crypta.lib.yt.PathUtils;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.transactions.utils.YtTransactionsUtils;
import ru.yandex.inside.yt.kosher.tables.YTableEntryType;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.transactions.Transaction;

@DisallowConcurrentExecution
public class DumpDatabaseJob implements Job {

    private static final Logger LOG = LoggerFactory.getLogger(DumpDatabaseJob.class);

    private final Paths paths;
    private final PostgresClient database;
    private final Yt yt;

    @Inject
    public DumpDatabaseJob(EEnvironment environment, YtService ytService, PostgresClient database) {
        this.paths = new Paths(YPath.simple("//home/crypta").child(PathUtils.toPath(environment)).child("lab"));
        this.database = database;
        this.yt = ytService.getHahn();
    }

    @Override
    @SuppressWarnings("unchecked")
    public void execute(JobExecutionContext context) throws JobExecutionException {
        LOG.info("Dumping database");
        withYtTransaction((transaction) -> {
            Option<GUID> transactionId = Option.of(transaction.getId());
            tablesToDump().forEach((GenericTable table) -> {
                Class entryClass = table.getEntryClass();
                YTableEntryType<Message> yEntryType = YTableEntryTypes.proto(entryClass);
                ListF<Message> entries = Cf.x(table.selectQuery().fetchInto(entryClass));
                Solomon.REGISTRY.gaugeInt64("lab.entity." + entryClass.getSimpleName()).set(entries.size());
                YPath path = paths.database().child(entryClass.getSimpleName());
                yt.tables().write(transactionId.toOptional(), false, path, yEntryType, entries.iterator());
            });
        });
    }

    private Tables tables() {
        return new Tables(database.getJooqConfiguration(), new ServiceSecurityContext());
    }

    private Collection<GenericTable> tablesToDump() {
        return Lists.newArrayList(
                tables().audiences(),
                tables().models(),
                tables().modelSegmentRelations(),
                tables().rules(),
                tables().rulesConditions(),
                tables().samples(),
                tables().segments(),
                tables().segmentExports()
        );
    }

    private void withYtTransaction(Consumer<Transaction> body) {
        try {
            YtTransactionsUtils.withTransaction(yt, Duration.ofMinutes(5), body::accept);
        } catch (RuntimeException e) {
            throw Exceptions.unchecked(e);
        }
    }
}
