package ru.yandex.crypta.lab.job;

import java.util.Map;
import java.util.Objects;
import java.util.stream.Stream;

import javax.inject.Inject;

import org.quartz.DisallowConcurrentExecution;
import org.quartz.Job;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.crypta.lab.LabService;
import ru.yandex.crypta.lab.SegmentService;
import ru.yandex.crypta.lab.proto.Segment;
import ru.yandex.crypta.lib.proto.EEnvironment;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeField;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeObject;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;

@DisallowConcurrentExecution
public class DumpSegmentsBuchhalterJob implements Job {

    private static final Map<EEnvironment, YPath> BUCHHALTER_GROUPS_LAB = Map.of(
            EEnvironment.ENV_PRODUCTION,
            YPath.simple("//home/crypta/production/buchhalter/audience/per_group/groups/lab")
    );

    private static final Logger LOG = LoggerFactory.getLogger(DumpSegmentsBuchhalterJob.class);

    private final Yt yt;
    private final SegmentService segmentService;
    private final YPath destination;

    @Inject
    public DumpSegmentsBuchhalterJob(YtService yt, LabService labService, EEnvironment environment) {
        this.yt = yt.getHahn();
        this.segmentService = labService.segments();
        this.destination = BUCHHALTER_GROUPS_LAB.getOrDefault(environment, null);
    }

    @Override
    @SuppressWarnings("StreamToIterable")
    public void execute(JobExecutionContext context) throws JobExecutionException {
        if (Objects.isNull(destination)) {
            LOG.info("Skipping dumping tags for buchhalter");
            return;
        }
        var records = segmentService
                .getAll()
                .stream()
                .flatMap(DumpSegmentsBuchhalterJob::getExports)
                .filter(DumpSegmentsBuchhalterJob::isAudienceExport)
                .flatMap(DumpSegmentsBuchhalterJob::toGroupRecords);
        yt.tables().write(destination, YTableEntryTypes.yson(GroupRecord.class), records::iterator);
    }

    private static Stream<Segment.Export> getExports(Segment segment) {
        return segment.getExports().getExportsList().stream();
    }

    private static boolean isAudienceExport(Segment.Export export) {
        return export.getKeywordId() == 557L;
    }

    private static Stream<GroupRecord> toGroupRecords(Segment.Export export) {
        return export.getTagsList().stream().map(tag -> GroupRecord.of(export.getSegmentId(), tag));
    }

    @YTreeObject
    private static class GroupRecord {

        @YTreeField(key="SegmentId")
        private long segmentId;

        @YTreeField(key="Group")
        private String group;

        public GroupRecord() {
        }

        public GroupRecord(long segmentId, String group) {
            this.segmentId = segmentId;
            this.group = group;
        }

        public long getSegmentId() {
            return segmentId;
        }

        public String getGroup() {
            return group;
        }

        public static GroupRecord of(long id, String tag) {
            return new GroupRecord(id, tag);
        }

    }
}
