package ru.yandex.crypta.lab.job;

import java.io.StringWriter;
import java.time.Duration;
import java.util.HashMap;

import javax.inject.Inject;

import NGetoptPb.Confoption;
import freemarker.template.Configuration;
import freemarker.template.TemplateExceptionHandler;
import org.apache.commons.mail.DefaultAuthenticator;
import org.apache.commons.mail.EmailConstants;
import org.apache.commons.mail.EmailException;
import org.apache.commons.mail.HtmlEmail;
import org.quartz.JobExecutionContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

import ru.yandex.crypta.clients.sandbox.SandboxClient;
import ru.yandex.crypta.clients.sandbox.entity.CustomField;
import ru.yandex.crypta.clients.sandbox.entity.Notification;
import ru.yandex.crypta.clients.sandbox.entity.Owner;
import ru.yandex.crypta.clients.sandbox.entity.Priority;
import ru.yandex.crypta.clients.sandbox.entity.TaskDraft;
import ru.yandex.crypta.clients.sandbox.entity.Type;
import ru.yandex.crypta.common.ws.auth.ServiceSecurityContext;
import ru.yandex.crypta.lab.LabService;
import ru.yandex.crypta.lib.proto.EEnvironment;
import ru.yandex.crypta.lib.proto.TSmtpConfig;

public abstract class LabJob extends SandboxJob {

    protected static final Logger LOG = LoggerFactory.getLogger(LabJob.class);

    public static final String SAMPLE_ID = "sample_id";

    @Inject
    private EEnvironment environment;

    @Inject
    private TSmtpConfig smtpConfig;

    @Inject
    private LabService lab;

    protected LabService lab() {
        return lab.withSecurityContext(new ServiceSecurityContext());
    }

    protected LabJob(SandboxClient sandbox) {
        super(sandbox);
    }

    protected TaskDraft defaultAugmentedTaskDraft(JobExecutionContext context) {
        String environment = getEnvironment(context);

        TaskDraft draft = createTaskDraft();
        draft.setType(defaultType());
        draft.setOwner(defaultOwner());
        draft.setPriority(defaultPriority());
        draft.setKillTimeout(Duration.ofHours(9).getSeconds());
        draft.addCustomField(new CustomField("task_name", taskName()));
        draft.addCustomField(new CustomField("environment", environment));
        draft.addCustomField(new CustomField("CLICKHOUSE_PASSWORD", "CRYPTA_CLICKHOUSE_PASSWORD"));
        draft.addCustomField(new CustomField("CRYPTA_API_TOKEN", "CRYPTA_API_TOKEN"));
        draft.addTag(tag(taskName()));
        draft.addTag(tag("crypta/api/" + environment));
        draft.addNotification(new Notification("email")
                .addRecipient("crypta-lab-sandbox")
                .addStatus("BREAK"));
        return draft;
    }

    protected Priority defaultPriority() {
        return new Priority("SERVICE", "NORMAL");
    }

    protected Owner defaultOwner() {
        return new Owner("CRYPTA");
    }

    protected Type defaultType() {
        return new Type("CRYPTA_LAB");
    }

    @Override
    protected void onFailure(JobExecutionContext context) {
        try {
            var configuration = createFreemarkerConfiguration();
            var template = configuration.getTemplate("task_failed.html");
            var emailBody = new StringWriter();
            var root = new HashMap<>();
            var sampleId = getArgument(context, SAMPLE_ID);
            MDC.put("sampleId", sampleId);
            var env = environment.getValueDescriptor().getOptions().getExtension(Confoption.val).toLowerCase();

            var sample = lab.samples().getSample(sampleId);

            root.put("sampleId", sampleId);
            root.put("environment", env);
            template.process(root, emailBody);

            var email = createEmail();
            email.setSubject("Крипта.Лаборатория: проблема с выборкой " + sampleId);
            email.setCharset(EmailConstants.UTF_8);
            email.setHtmlMsg(emailBody.toString());
            email.addTo(yandexTeam(sample.getAuthor()));
            email.addCc(yandexTeam("crypta-lab-sandbox"));

            email.send();
        } catch (Exception e) {
            LOG.error("Failed to send email", e);
        }
    }

    private Configuration createFreemarkerConfiguration() {
        var configuration = new Configuration(Configuration.VERSION_2_3_28);
        configuration.setClassForTemplateLoading(LabJob.class, "/emails");
        configuration.setTemplateExceptionHandler(TemplateExceptionHandler.RETHROW_HANDLER);
        configuration.setDefaultEncoding("UTF-8");
        configuration.setWrapUncheckedExceptions(true);
        return configuration;
    }

    protected HtmlEmail createEmail() throws EmailException {
        var email = new HtmlEmail();
        email.setHostName(smtpConfig.getServer());
        email.setSmtpPort(smtpConfig.getPort());
        email.setAuthenticator(new DefaultAuthenticator(smtpConfig.getUsername(), smtpConfig.getPassword()));
        email.setSSLOnConnect(smtpConfig.getUseSsl());
        email.setFrom(yandexTeam(smtpConfig.getUsername()));
        return email;
    }

    protected String yandexTeam(String username) {
        return username + "@yandex-team.ru";
    }

    abstract protected String taskName();

}
