package ru.yandex.crypta.lab.job;

import java.time.Duration;
import java.time.Instant;
import java.util.Collections;
import java.util.Date;
import java.util.Objects;
import java.util.UUID;

import org.quartz.JobDetail;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.quartz.JobKey;
import org.quartz.SchedulerException;
import org.quartz.Trigger;
import org.quartz.TriggerKey;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.crypta.common.ws.solomon.Solomon;
import ru.yandex.monlib.metrics.primitives.Rate;

public abstract class RetryingJob extends CommonJob {

    public enum ExecuteResult {
        SUCCESS,
        RECOVERABLE_FAILURE,
        UNRECOVERABLE_FAILURE,
        CHECK_AGAIN_LATER
    }

    public static final String RETRIES_LEFT = "_retries_left";

    private static final Logger LOG = LoggerFactory.getLogger(RetryingJob.class);

    private static final Rate FAILURES_RATE = Solomon.REGISTRY.rate("lab.job.failure");
    private static final Rate SUCCESS_RATE = Solomon.REGISTRY.rate("lab.job.success");
    private static final Rate RETRIES_RATE = Solomon.REGISTRY.rate("lab.job.retry");

    @Override
    public void execute(JobExecutionContext context) throws JobExecutionException {
        try {
            maybeSetInitialRetriesLeft(context);
            if (!hasRetriesLeft(context)) {
                FAILURES_RATE.inc();
                onFailure(context);
                throwRetriesExceeded();
            }
            ExecuteResult executeResult = executeSafe(context);
            if (Objects.equals(executeResult, ExecuteResult.SUCCESS)) {
                SUCCESS_RATE.inc();
                onSuccess(context);
            }
        } catch (JobExecutionException jobException) {
            throw jobException;
        } catch (Throwable throwable) {
            LOG.error("Task {} failed", context.getJobDetail().getKey(), throwable);
            RETRIES_RATE.inc();
            decrementRetriesLeft(context);
            onRetry(context);
            reschedule(context, getRetryInterval());
        }
    }

    protected void onSuccess(JobExecutionContext context) {
        LOG.info("Task {} complete", context.getJobDetail().getKey());
    }

    protected void onFailure(JobExecutionContext context) {
        LOG.info("Task {} failed with no attempts left", context.getJobDetail().getKey());
    }

    protected void onRetry(JobExecutionContext context) {
        LOG.info("Task {} failed, will retry", context.getJobDetail().getKey());
    }

    protected void throwRetriesExceeded() throws JobExecutionException {
        LOG.error("No retries left");
        JobExecutionException jobException = new JobExecutionException("Maximum number of retries exceeded");
        jobException.setRefireImmediately(false);
        jobException.setUnscheduleFiringTrigger(true);
        throw jobException;
    }

    protected void reschedule(JobExecutionContext context, Duration interval) throws JobExecutionException {
        Date nextFireTime = Date.from(Instant.now().plus(interval));
        TriggerKey triggerKey = context.getTrigger().getKey();
        JobKey jobKey = context.getJobDetail().getKey();
        String name = UUID.randomUUID().toString();

        JobDetail jobDetail = context.getJobDetail()
                .getJobBuilder()
                .withIdentity(name, jobKey.getGroup())
                .build();
        Trigger newTrigger =
                context.getTrigger().getTriggerBuilder()
                        .withIdentity(name, triggerKey.getGroup())
                        .withDescription("Retry of " + triggerKey.toString())
                        .startAt(nextFireTime)
                        .forJob(jobDetail)
                        .usingJobData(context.getMergedJobDataMap())
                        .build();
        try {
            context.getScheduler().scheduleJob(jobDetail, Collections.singleton(newTrigger), true);
            LOG.info("Rescheduled task {}", jobKey);
        } catch (SchedulerException e) {
            throw new JobExecutionException(e);
        }
    }

    private int getRetriesLeft(JobExecutionContext context) {
        return context.getJobDetail().getJobDataMap().getInt(RETRIES_LEFT);
    }

    private boolean hasRetriesLeft(JobExecutionContext context) {
        int retriesLeft = getRetriesLeft(context);
        LOG.info("Task {} has {} more retries", context.getJobDetail().getKey(), retriesLeft);
        return retriesLeft > 0;
    }

    protected void decrementRetriesLeft(JobExecutionContext context) {
        putArgument(context, RETRIES_LEFT, getRetriesLeft(context) - 1);
    }

    private void maybeSetInitialRetriesLeft(JobExecutionContext context) {
        if (!hasArgument(context, RETRIES_LEFT)) {
            putArgument(context, RETRIES_LEFT, getInitialRetriesCount());
        }
    }

    protected int getInitialRetriesCount() {
        return 100;
    }

    protected Duration getRetryInterval() {
        return Duration.ofMinutes(10);
    }

    protected abstract ExecuteResult executeSafe(JobExecutionContext context) throws JobExecutionException;

}
