package ru.yandex.crypta.lab.migrations;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import org.flywaydb.core.api.migration.BaseJavaMigration;
import org.flywaydb.core.api.migration.Context;
import org.jooq.DSLContext;
import org.jooq.Query;
import org.jooq.Record3;
import org.jooq.Result;
import org.jooq.impl.DSL;

import ru.yandex.crypta.clients.pgaas.PostgresClient;
import ru.yandex.crypta.common.ws.auth.ServiceSecurityContext;
import ru.yandex.crypta.lab.Identifier;
import ru.yandex.crypta.lab.proto.EHashingMethod;
import ru.yandex.crypta.lab.proto.ELabIdentifierType;
import ru.yandex.crypta.lab.proto.Sample;
import ru.yandex.crypta.lab.proto.TSampleView;
import ru.yandex.crypta.lab.proto.TSampleViewOptions;
import ru.yandex.crypta.lab.tables.SampleViewsTable;
import ru.yandex.crypta.lab.tables.SamplesTable;
import ru.yandex.crypta.lab.utils.Proto;

@SuppressWarnings({"TypeName"})
public class R__Restore_type_views extends BaseJavaMigration {

    public static final String YANDEXUID = "yandexuid";

    @Override
    public Integer getChecksum() {
        return 1;
    }

    @Override
    public void migrate(Context context) throws Exception {
        var connection = context.getConfiguration().getDataSource().getConnection();
        DSLContext dsl = DSL.using(connection);

        ServiceSecurityContext securityContext = new ServiceSecurityContext();
        SamplesTable samplesTable =
                new SamplesTable(PostgresClient.wrap(connection), securityContext);
        SampleViewsTable sampleViewsTable =
                new SampleViewsTable(PostgresClient.wrap(connection), securityContext);

        Result<Record3<String, String, byte[]>> sampleViews = dsl
                .select(SampleViewsTable.SAMPLE_ID, SampleViewsTable.ID, SampleViewsTable.OPTIONS)
                .from(SampleViewsTable.TABLE)
                .fetch();

        List<Record3<String, String, byte[]>> broken = sampleViews.stream()
                .filter(each -> {
                    TSampleViewOptions options =
                            Proto.readOrDefault(each.value3(), TSampleViewOptions.parser(),
                                    TSampleViewOptions.getDefaultInstance());
                    return options.getMatching().getIdType().equals(ELabIdentifierType.LAB_ID_UNKNOWN);
                })
                .collect(Collectors.toList());

        broken.forEach(each -> {
            String sampleId = each.value1();
            String viewId = each.value2();

            Sample sample = samplesTable.selectByIdQuery(sampleId).fetchOneInto(Sample.class);
            TSampleView brokenView = sampleViewsTable.selectByIdQuery(sampleId, viewId).fetchOneInto(TSampleView.class);

            TSampleView.Builder view = TSampleView.newBuilder()
                    .setPath(brokenView.getPath())
                    .setState(brokenView.getState())
                    .setID(viewId)
                    .setSampleID(sampleId);

            if (Objects.equals(viewId, YANDEXUID)) {
                view.getOptionsBuilder()
                        .getMatchingBuilder()
                        .setIdType(ELabIdentifierType.LAB_ID_YANDEXUID)
                        .setKey("yuid")
                        .setHashingMethod(EHashingMethod.HM_IDENTITY)
                        .setIncludeOriginal(true);
            } else {
                Identifier identifier = Identifier.byName(sample.getIdName());
                view.getOptionsBuilder()
                        .getMatchingBuilder()
                        .setIdType(identifier.getLabType())
                        .setKey(sample.getIdKey())
                        .setHashingMethod(identifier.getHashingMethod())
                        .setIncludeOriginal(true);
            }
            Query query = sampleViewsTable.insertOrUpdateQuery(view.build());
            query.execute();
        });
    }
}
