package ru.yandex.crypta.lab.rule_estimates;

import java.time.Duration;
import java.util.List;
import java.util.Optional;

import javax.inject.Inject;

import io.grpc.ManagedChannelBuilder;
import io.grpc.StatusRuntimeException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.crypta.lab.rule_estimator.proto.RuleEstimateStats;
import ru.yandex.crypta.lab.rule_estimator.proto.Update;
import ru.yandex.crypta.lab.rule_estimator.services.api.proto.GetRuleConditionStatsRequest;
import ru.yandex.crypta.lab.rule_estimator.services.api.proto.GetRuleStatsRequest;
import ru.yandex.crypta.lab.rule_estimator.services.api.proto.RuleEstimatorGrpc;
import ru.yandex.crypta.lab.rule_estimator.services.api.proto.RuleEstimatorGrpc.RuleEstimatorBlockingStub;
import ru.yandex.crypta.lab.rule_estimator.services.api.proto.UpdateResponse;
import ru.yandex.crypta.lab.utils.GrpcRetries;
import ru.yandex.crypta.lib.proto.TRuleEstimatorConfig;

public class RuleEstimatorClient {
    private final RuleEstimatorBlockingStub blockingStub;
    private static final Logger LOG = LoggerFactory.getLogger(RuleEstimatorClient.class);

    @Inject
    public RuleEstimatorClient(TRuleEstimatorConfig config) {
        var channel = ManagedChannelBuilder.forAddress(config.getHost(), config.getPort())
                .defaultServiceConfig(
                    GrpcRetries.buildConfig(4D, Duration.ofMillis(50), Duration.ofSeconds(1), 2D)
                )
                .enableRetry()
                .usePlaintext()
                .build();
        blockingStub = RuleEstimatorGrpc.newBlockingStub(channel);
    }

    public Optional<RuleEstimateStats> getRuleConditionStats(long ruleConditionId) {
        try {
            return Optional.of(blockingStub.getRuleConditionStats(GetRuleConditionStatsRequest.newBuilder().setRuleConditionId(ruleConditionId).build()));
        } catch (StatusRuntimeException e) {
            LOG.warn("RPC failed: {}", e.getStatus());
            return Optional.empty();
        }
    }

    public Optional<RuleEstimateStats> getRuleStats(List<Long> ruleConditionIds) {
        try {
            return Optional.of(blockingStub.getRuleStats(GetRuleStatsRequest.newBuilder().addAllRuleConditionIds(ruleConditionIds).build()));
        } catch (StatusRuntimeException e) {
            LOG.warn("RPC failed: {}", e.getStatus());
            return Optional.empty();
        }
    }

    public Optional<UpdateResponse> update(String ruleId, List<Long> ruleConditionIds) {
        try {
            return Optional.of(blockingStub.update(Update.newBuilder().addAllRuleConditionIds(ruleConditionIds).setRuleId(ruleId).build()));
        } catch (StatusRuntimeException e) {
            LOG.warn("RPC failed: {}", e.getStatus());
            return Optional.empty();
        }
    }
}
