package ru.yandex.crypta.lab.siberia;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.TimeUnit;

import javax.inject.Inject;

import okhttp3.HttpUrl;
import okhttp3.OkHttpClient;
import okhttp3.ResponseBody;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import retrofit2.Call;
import retrofit2.Retrofit;

import ru.yandex.crypta.clients.tvm.TvmClient;
import ru.yandex.crypta.clients.tvm.TvmOkHttpInterceptor;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.lib.proto.TSiberiaConfig;

public class SiberiaClient {

    private static final Logger LOG = LoggerFactory.getLogger(SiberiaClient.class);

    private final TSiberiaConfig config;
    private final TvmClient tvm;

    @Inject
    public SiberiaClient(TSiberiaConfig config, TvmClient tvm) {
        this.config = config;
        this.tvm = tvm;
        this.tvm.setDstClientIds(List.of(config.getTvm().getDestinationTvmId()));
    }

    public SiberiaApi get() {
        var client = new OkHttpClient.Builder()
                .addInterceptor(chain -> {
                    var request = chain.request();
                    LOG.info("Call Siberia {}", request.url());
                    return chain.proceed(request);
                })
                .addInterceptor(new TvmOkHttpInterceptor(tvm, getDestinationTvm()))
                .readTimeout(120, TimeUnit.SECONDS)
                .build();
        HttpUrl url = new HttpUrl.Builder().scheme("http").host(getSiberiaHost()).port(80).build();
        var retrofit = new Retrofit.Builder().baseUrl(url).client(client).build();
        return retrofit.create(SiberiaApi.class);
    }

    public RawResponse tryCall(SiberiaCaller callback) {
        try {
            var response = callback.call(get()).execute();

            if (response.code() == 404) {
                throw Exceptions.notFound();
            }

            if (response.code() == 405) {
                throw Exceptions.wrongRequestException("Not allowed", "NOT_READY");
            }

            if (response.isSuccessful()) {
                return new RawResponse(response.body().string(), response.code());
            } else {
                return new RawResponse(response.errorBody().string(), response.code());
            }
        } catch (IOException e) {
            throw Exceptions.internal("Error while sending HTTP request to Siberia: " + e.getMessage());
        }
    }

    private String getSiberiaHost() {
        return config.getHost();
    }

    private int getDestinationTvm() {
        return config.getTvm().getDestinationTvmId();
    }

    public interface SiberiaCaller {
        Call<ResponseBody> call(SiberiaApi siberia) throws IOException;
    }

}
