package ru.yandex.crypta.lab.siberia;

import java.io.IOException;
import java.util.Optional;

import com.google.protobuf.util.JsonFormat;

import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.siberia.core.proto.TAddUserSetResponse;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

public class SiberiaHelpers {
    public static final String ROW_COUNT = "row_count";
    private static final long MAX_SIBERIA_USER_SET_ROW_COUNT = 50 * 1000 * 1000;

    static public RawResponse removeSiberiaUserSet(SiberiaClient siberiaClient, String userSetId) {
        return siberiaClient.tryCall(siberia -> siberia.deleteUserSet(userSetId));
    }

    static public Optional<String> tryCreateSiberiaUserSetForTable(SiberiaClient siberiaClient, YTreeNode table, String title, Long ttl) {
        try {
            validateTableForSiberiaUserSet(table);
            return Optional.of(createSiberiaUserSet(siberiaClient, title, ttl));
        } catch (Exception e) {
            return Optional.empty();
        }
    }

    static private void validateTableForSiberiaUserSet(YTreeNode table) {
        if (table.getAttributeOrThrow(ROW_COUNT).longValue() > MAX_SIBERIA_USER_SET_ROW_COUNT) {
            throw Exceptions.wrongRequestException("Too big table", "TABLE_IS_TOO_BIG");
        }
    }

    static private String createSiberiaUserSet(SiberiaClient siberiaClient, String title, Long ttl) {
        var response = siberiaClient.tryCall(siberia -> siberia.createUserSet(title, ttl));

        var responseCode = response.getHttpCode();
        if (responseCode != 200) {
            throw Exceptions.internal(
                    "user_sets/add Siberia handle return code = '" + responseCode + "' and message = '"
                            + response.getText() + "'");
        }

        try {
            var builder = TAddUserSetResponse.newBuilder();
            JsonFormat.parser().merge(response.getText(), builder);
            return builder.build().getUserSetId();
        } catch (IOException e) {
            throw Exceptions.internal("Error while parsing response from Siberia: " + e.getMessage());
        }
    }
}
