package ru.yandex.crypta.lab.startrek;

import java.util.List;
import java.util.stream.Collectors;

import javax.inject.Inject;

import ru.yandex.crypta.clients.pgaas.PostgresClient;
import ru.yandex.crypta.clients.startrek.CryptaStartrekClient;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.common.exception.NotFoundException;
import ru.yandex.crypta.lab.LabService;
import ru.yandex.crypta.lab.base.BaseService;
import ru.yandex.crypta.lab.proto.Rule;
import ru.yandex.crypta.lab.proto.RuleCondition;
import ru.yandex.crypta.lab.tables.RulesConditionsTable;
import ru.yandex.crypta.lib.proto.EEnvironment;
import ru.yandex.startrek.client.model.CommentCreate;
import ru.yandex.startrek.client.model.Issue;
import ru.yandex.startrek.client.model.IssueCreate;
import ru.yandex.startrek.client.model.IssueUpdate;
import ru.yandex.startrek.client.model.Relationship;

public class DefaultLabStartrekService extends BaseService<DefaultLabStartrekService> implements LabStartrekService {
    private final static String APPROVE_RULE_QUEUE = "CRYPTA";
    private final static long APPROVE_RULE_COMPONENT_ID = 79327L;
    private final static String SERVICE_REQUEST_TASK_TYPE = "serviceRequest";
    private final static String RULE_APPROVAL_REQUEST_SUMMARY = "Проверка правил для";

    private final CryptaStartrekClient startrek;
    private final LabService lab;

    @Inject
    public DefaultLabStartrekService(
            CryptaStartrekClient startrek,
            EEnvironment environment,
            PostgresClient sql,
            LabService lab
    ) {
        super(environment, sql);
        this.startrek = startrek;
        this.lab = lab;
    }

    @Override
    public Issue createApproveSegmentRuleIssue(
            String summary, String description, String linkedIssue, String login, String url
    ) {
        if (summary == null || description == null || login == null ) {
            throw Exceptions.wrongRequestException(
                    "One or more parameters are missing: summary=" + summary
                            + "; description=" + description + "; login=" + login,
                    "MISSING_PARAMETER"
            );
        }

        IssueCreate.Builder create = IssueCreate.builder()
                .queue(APPROVE_RULE_QUEUE)
                .summary(summary)
                .type(SERVICE_REQUEST_TASK_TYPE)
                .components(startrek.get().components().get(APPROVE_RULE_COMPONENT_ID))
                .description(String.join("\n", url, "\n",description))
                .author(login);

        if (linkedIssue != null && !linkedIssue.equals("")) {
            create = create.link(linkedIssue, Relationship.RELATES);
        }

        return startrek.createIssue(create.build());
    }

    @Override
    public Issue commentOnApproveSegmentRuleIssue(String issueId, String comment) {
        if (issueId == null || comment == null ) {
            throw Exceptions.wrongRequestException(
                    "One or more parameters are missing: issueId=" + issueId + "; comment=" + comment,
                    "MISSING_PARAMETER"
            );
        }

        Issue issue = startrek.getIssueById(issueId);

        IssueUpdate.Builder update = IssueUpdate.builder()
                .comment(CommentCreate.comment(comment).build());

        return issue.update(update.build());
    }

    @Override
    public Rule requestRuleApprovalViaTicket(
            String ruleId, String login, String linkedIssue, String url
    ) {
        return withSqlTransaction(tables -> {
            Rule rule = tables().rules()
                    .selectByIdQuery(ruleId)
                    .fetchOptionalInto(Rule.class)
                    .orElseThrow(NotFoundException::new);

            List<RuleCondition> unapprovedConditions = tables().rulesConditions().selectUnapprovedByRuleIdQuery(ruleId)
                    .fetch(RulesConditionsTable::readCondition);

            String issueId = rule.getIssueId();
            String segmentName = rule.getName();
            Long status = null;

            if (!issueId.equals("")) {
                Issue issue = startrek.getIssueById(issueId);

                if (issue != null) {
                    status = issue.getStatus().getId();
                }
            }

            List<String> descriptions = unapprovedConditions.stream().map(condition ->
                    "**" + condition.getSource().toString() + "**"
                            + "\n\n" + String.join("\n", condition.getValuesList())).collect(Collectors.toList());

            String comment = String.join("\n\n", descriptions);

            if (issueId.equals("") || (status != null && status.equals(IssueStatus.CLOSED.id()))) {

                Issue newIssue = createApproveSegmentRuleIssue(
                        RULE_APPROVAL_REQUEST_SUMMARY + " " + segmentName,
                        comment,
                        linkedIssue,
                        login,
                        url
                );

                tables().rules().updateQuery(
                        ruleId,
                        rule.toBuilder()
                                .setIssueId(newIssue.getId())
                                .setIssueKey(newIssue.getKey())
                                .build()
                ).execute();
            } else {
                commentOnApproveSegmentRuleIssue(issueId, comment);
            }

            return tables().rules()
                    .selectByIdQuery(ruleId)
                    .fetchOptionalInto(Rule.class)
                    .orElseThrow(NotFoundException::new);
        });
    }

    @Override
    public DefaultLabStartrekService clone() {
        return new DefaultLabStartrekService(startrek, environment(), sql(), lab);
    }
}
