package ru.yandex.crypta.lab.tables;

import javax.ws.rs.core.SecurityContext;

import org.jooq.Configuration;
import org.jooq.DeleteConditionStep;
import org.jooq.Field;
import org.jooq.Query;
import org.jooq.Record;
import org.jooq.Select;
import org.jooq.Table;
import org.jooq.Update;
import org.jooq.impl.DSL;

import ru.yandex.crypta.common.data.GenericTable;
import ru.yandex.crypta.lab.proto.Audience;
import ru.yandex.crypta.lab.proto.Timestamps;


public class AudiencesTable extends GenericTable<Audience> {

    public static final Table<Record> TABLE = DSL.table("api_audiences");

    public static final Field<String> ID = DSL.field(DSL.name(TABLE.getName(), "id"), String.class);
    public static final Field<String> AUTHOR = DSL.field(DSL.name(TABLE.getName(), "author"), String.class);
    public static final Field<String> NAME = DSL.field(DSL.name(TABLE.getName(), "name"), String.class);
    public static final Field<String> PURPOSE = DSL.field(DSL.name(TABLE.getName(), "purpose"), String.class);
    public static final Field<String> LOGIN = DSL.field(DSL.name(TABLE.getName(), "login"), String.class);
    public static final Field<Long> EXTERNAL_ID = DSL.field(DSL.name(TABLE.getName(), "external_id"), Long.class);
    public static final Field<String> SOURCE_PATH = DSL.field(DSL.name(TABLE.getName(), "source_path"), String.class);
    public static final Field<String> SOURCE_FIELD = DSL.field(DSL.name(TABLE.getName(), "source_field"), String.class);
    private static final Field<String> STATE = DSL.field(DSL.name(TABLE.getName(), "state"), String.class);
    private static final Field<Long> CREATED = DSL.field(DSL.name(TABLE.getName(), "created"), Long.class);
    private static final Field<Long> MODIFIED = DSL.field(DSL.name(TABLE.getName(), "modified"), Long.class);

    private final SecurityContext securityContext;

    public AudiencesTable(Configuration configuration, SecurityContext securityContext) {
        super(configuration, Audience.class);
        this.securityContext = securityContext;
    }

    @Override
    protected Audience read(Record record) {
        return Audience.newBuilder()
                .setId(record.get(ID))
                .setAuthor(record.get(AUTHOR))
                .setName(record.get(NAME))
                .setPurpose(record.get(PURPOSE))
                .setLogin(record.get(LOGIN))
                .setExternalId(record.get(EXTERNAL_ID))
                .setSourcePath(record.get(SOURCE_PATH))
                .setSourceField(record.get(SOURCE_FIELD))
                .setState(Audience.State.valueOf(record.get(STATE)))
                .setTimestamps(Timestamps.newBuilder()
                        .setCreated(record.get(CREATED))
                        .setModified(record.get(MODIFIED)))
                .build();
    }

    public Query insertQuery(Audience audience)
    {
        return genericInsertQuery(TABLE, audience);
    }

    private Query genericInsertQuery(Table<Record> table, Audience audience)
    {
        return dsl.insertInto(table)
                .set(ID, audience.getId())
                .set(AUTHOR, audience.getAuthor())
                .set(NAME, audience.getName())
                .set(PURPOSE, audience.getPurpose())
                .set(LOGIN, audience.getLogin())
                .set(EXTERNAL_ID, audience.getExternalId())
                .set(SOURCE_PATH, audience.getSourcePath())
                .set(SOURCE_FIELD, audience.getSourceField())
                .set(STATE, audience.getState().toString())
                .set(CREATED, audience.getTimestamps().getCreated())
                .set(MODIFIED, audience.getTimestamps().getModified());
    }

    @Override
    public Select<Record> selectQuery() {
        return dsl.selectFrom(TABLE)
                .where(AudiencesAcl.isAccessibleBy(securityContext));
    }

    public Select<Record> selectMineQuery() {
        return dsl.selectFrom(TABLE)
                .where(AudiencesAcl.isOf(securityContext));
    }

    public Select<Record> selectByIdAccessibleQuery(String id) {
        return dsl.selectFrom(TABLE)
                .where(ID.eq(id).and(AudiencesAcl.isAccessibleBy(securityContext)));
    }

    public Select<Record> selectByIdModifiableQuery(String id) {
        return dsl.selectFrom(TABLE)
                .where(ID.eq(id).and(AudiencesAcl.isModifiableBy(securityContext)));
    }

    public Update<Record> updateQuery(Audience audience) {
        return dsl.update(TABLE)
                .set(AUTHOR, audience.getAuthor())
                .set(NAME, audience.getName())
                .set(PURPOSE, audience.getPurpose())
                .set(LOGIN, audience.getLogin())
                .set(EXTERNAL_ID, audience.getExternalId())
                .set(SOURCE_PATH, audience.getSourcePath())
                .set(SOURCE_FIELD, audience.getSourceField())
                .set(STATE, audience.getState().toString())
                .set(CREATED, audience.getTimestamps().getCreated())
                .set(MODIFIED, audience.getTimestamps().getModified())
                .where(ID.eq(audience.getId()).and(AudiencesAcl.isModifiableBy(securityContext)));
    }

    public DeleteConditionStep<Record> deleteQuery(String id) {
        return dsl.deleteFrom(TABLE)
                .where(ID.eq(id).and(AudiencesAcl.isModifiableBy(securityContext)));
    }

}
