package ru.yandex.crypta.lab.tables;

import java.io.IOException;
import java.util.Map;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.jooq.Configuration;
import org.jooq.Field;
import org.jooq.Insert;
import org.jooq.Record;
import org.jooq.Select;
import org.jooq.Table;
import org.jooq.Update;
import org.jooq.impl.DSL;

import ru.yandex.crypta.common.data.GenericTable;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.lab.Keyword;


public class KeywordsTable extends GenericTable<Keyword> {

    private static final Table<Record> TABLE = DSL.table("api_keywords");

    private static final Field<String> ID = DSL.field(DSL.name(TABLE.getName(), "id"), String.class);
    private static final ObjectMapper MAPPER = new ObjectMapper();
    private static final Field<String> SEGMENTS = DSL.field(DSL.name(TABLE.getName(), "segment"), String.class);

    public KeywordsTable(Configuration configuration) {
        super(configuration, Keyword.class);
    }

    @Override
    protected Keyword read(Record record) {
        try {
            Map<String, String> segments =
                    MAPPER.readValue(record.get(SEGMENTS), new TypeReference<Map<String, String>>() {
                    });
            return new Keyword(record.get(ID), segments);
        } catch (IOException e) {
            throw Exceptions.unchecked(e);
        }
    }

    @Override
    public Select selectQuery() {
        return dsl.selectFrom(TABLE);
    }

    public Select<Record> selectByIdQuery(String id) {
        return dsl.selectFrom(TABLE).where(ID.equal(id));
    }

    public Insert<Record> insertById(Keyword keyword) {
        try {

            return dsl.insertInto(TABLE, ID, SEGMENTS)
                    .values(keyword.getId(), MAPPER.writeValueAsString(keyword.getSegments()));
        } catch (JsonProcessingException e) {
            throw Exceptions.unchecked(e);
        }
    }

    public Update<Record> updateById(Keyword keyword) {
        try {
            return dsl.update(TABLE)
                    .set(SEGMENTS, MAPPER.writeValueAsString(keyword.getSegments()))
                    .where(ID.eq(keyword.getId()));
        } catch (JsonProcessingException e) {
            throw Exceptions.unchecked(e);
        }
    }

}
