package ru.yandex.crypta.lab.tables;

import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

import org.jooq.Configuration;
import org.jooq.DeleteConditionStep;
import org.jooq.Field;
import org.jooq.Query;
import org.jooq.Record;
import org.jooq.Select;
import org.jooq.SelectHavingConditionStep;
import org.jooq.SelectHavingStep;
import org.jooq.SelectLimitPercentStep;
import org.jooq.Table;
import org.jooq.impl.DSL;

import ru.yandex.crypta.common.data.GenericTable;
import ru.yandex.crypta.lab.proto.Model;
import ru.yandex.crypta.lab.proto.Timestamps;

public class ModelsTable extends GenericTable<Model> {

    public static final Table<Record> TABLE = DSL.table("api_models");
    public static final Field<String> ID = DSL.field(DSL.name(TABLE.getName(), "id"), String.class);
    private static final Field<String> DESCRIPTION = DSL.field(DSL.name(TABLE.getName(), "description"), String.class);
    private static final Field<String> URI = DSL.field(DSL.name(TABLE.getName(), "uri"), String.class);
    private static final Field<String> S3_URI = DSL.field(DSL.name(TABLE.getName(), "s3_uri"), String.class);
    private static final Field<String> TAG = DSL.field(DSL.name(TABLE.getName(), "tag"), String.class);
    private static final Field<String> STATE = DSL.field(DSL.name(TABLE.getName(), "state"), String.class);
    private static final Field<Long> CREATED = DSL.field(DSL.name(TABLE.getName(), "created"), Long.class);
    private static final Field<Long> MODIFIED = DSL.field(DSL.name(TABLE.getName(), "modified"), Long.class);

    public ModelsTable(Configuration configuration) {
        super(configuration, Model.class);
    }

    @Override
    public Model read(Record record) {
        String modelId = record.get(ID);
        Model.Builder builder = Model
                .newBuilder()
                .setId(modelId)
                .setDescription(record.get(DESCRIPTION))
                .setUri(record.get(URI))
                .setS3Uri(Optional.ofNullable(record.get(S3_URI)).orElse(""))
                .setTag(Optional.ofNullable(record.get(TAG)).orElse(""))
                .setState(Model.State.valueOf(record.get(STATE)))
                .setTimestamps(Timestamps.newBuilder()
                        .setCreated(record.get(CREATED))
                        .setModified(record.get(MODIFIED)));
        ModelSegmentRelationsTable.readAggregated(record).forEachOrdered(builder::addSegments);
        return builder.build();
    }

    public SelectHavingConditionStep<Record> selectByIdQuery(String id) {
        return selectQuery().having(ID.eq(id));
    }

    public DeleteConditionStep<Record> deleteByIdQuery(String id) {
        return dsl.deleteFrom(TABLE).where(ID.equal(id));
    }

    public Query insertQuery(Model model) {
        return genericInsertQuery(TABLE, model);
    }

    private Query genericInsertQuery(Table<Record> table, Model model) {
        return dsl.insertInto(table)
                .set(ID, model.getId())
                .set(DESCRIPTION, model.getDescription())
                .set(URI, model.getUri())
                .set(S3_URI, model.getS3Uri())
                .set(TAG, model.getTag())
                .set(STATE, model.getState().name())
                .set(MODIFIED, model.getTimestamps().getModified())
                .set(CREATED, model.getTimestamps().getCreated());
    }

    public Query updateQuery(Model model) {
        return dsl.update(TABLE)
                .set(DESCRIPTION, model.getDescription())
                .set(URI, model.getUri())
                .set(S3_URI, model.getS3Uri())
                .set(TAG, model.getTag())
                .set(STATE, model.getState().name())
                .set(MODIFIED, model.getTimestamps().getModified())
                .set(CREATED, model.getTimestamps().getCreated())
                .where(ID.eq(model.getId()));
    }

    @Override
    public SelectHavingStep<Record> selectQuery() {
        return dsl.select(selectFields()).from(
                TABLE.leftJoin(ModelSegmentRelationsTable.TABLE)
                        .on(ID.eq(ModelSegmentRelationsTable.MODEL_ID))
        ).groupBy(ID);
    }

    public SelectLimitPercentStep<Record> selectByTagQuery(String tag) {
        return selectQuery().having(TAG.eq(tag)).orderBy(MODIFIED).limit(1);
    }

    public Select<Record> selectByIdsQuery(List<String> ids) {
        return selectQuery().having(ID.in(ids));
    }

    private Field[] selectFields() {
        Field[] tableFields = {
                ID,
                DESCRIPTION,
                URI,
                S3_URI,
                TAG,
                STATE,
                CREATED,
                MODIFIED,
        };
        Field[] foreignFields = ModelSegmentRelationsTable.aggregateFields();
        return Stream.concat(Stream.of(tableFields), Stream.of(foreignFields)).toArray(Field[]::new);
    }
}
