package ru.yandex.crypta.lab.tables;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import javax.ws.rs.core.SecurityContext;

import com.google.common.collect.ImmutableMap;
import org.jooq.Condition;
import org.jooq.Configuration;
import org.jooq.DeleteConditionStep;
import org.jooq.Field;
import org.jooq.Query;
import org.jooq.Record;
import org.jooq.Select;
import org.jooq.Sequence;
import org.jooq.Table;
import org.jooq.Update;
import org.jooq.impl.DSL;

import ru.yandex.crypta.common.data.GenericTable;
import ru.yandex.crypta.lab.formatters.AppFormatter;
import ru.yandex.crypta.lab.formatters.IdentityFormatter;
import ru.yandex.crypta.lab.formatters.IntWithCommentFormatter;
import ru.yandex.crypta.lab.formatters.MetricaCounterFormatter;
import ru.yandex.crypta.lab.formatters.RuleConditionFormatter;
import ru.yandex.crypta.lab.formatters.RuleFormatUtils;
import ru.yandex.crypta.lab.formatters.UrlFormatter;
import ru.yandex.crypta.lab.formatters.WordRuleFormatter;
import ru.yandex.crypta.lab.proto.RuleCondition;
import ru.yandex.crypta.lab.proto.Timestamps;


public class RulesConditionsTable extends GenericTable<RuleCondition> {

    public static final Table<Record> TABLE = DSL.table("api_constructor_rules_conditions");

    public static final Field<String> RULE_ID = DSL.field(DSL.name(TABLE.getName(), "rule_id"), String.class);
    public static final Field<String> SOURCE = DSL.field(DSL.name(TABLE.getName(), "source"), String.class);
    public static final Field<String> STATE = DSL.field(DSL.name(TABLE.getName(), "state"), String.class);
    public static final Field<String> VALUE = DSL.field(DSL.name(TABLE.getName(), "value"), String.class);
    public static final Field<Long> REVISION = DSL.field(DSL.name(TABLE.getName(), "revision"), Long.class);
    private static final Field<Long> CREATED = DSL.field(DSL.name(TABLE.getName(), "created"), Long.class);
    private static final Field<Long> MODIFIED = DSL.field(DSL.name(TABLE.getName(), "modified"), Long.class);
    public static final Field<Boolean> HAS_ERRORS =
            DSL.field(DSL.name(TABLE.getName(), "has_errors"), Boolean.class);

    public static final Sequence<Long> REVISION_SEQ =
            DSL.sequence(DSL.name("api_constructor_rules_conditions_revision_seq"), Long.class);

    private static final Field<String[]> RULE_IDS = DSL.arrayAgg(RULE_ID).as("rule_ids");
    private static final Field<String[]> SOURCES = DSL.arrayAgg(SOURCE).as("sources");
    public static final Field<String[]> STATES = DSL.arrayAgg(STATE).as("states");
    private static final Field<String[]> VALUES = DSL.arrayAgg(VALUE).as("values");
    private static final Field<Long[]> REVISIONS = DSL.arrayAgg(REVISION).as("revisions");
    private static final Field<Long[]> CREATED_TIMESTAMPS = DSL.arrayAgg(CREATED).as("created_timestamps");
    private static final Field<Long[]> MODIFIED_TIMESTAMPS = DSL.arrayAgg(MODIFIED).as("modified_timestamps");
    public static final Field<Boolean[]> HAS_ERRORS_VALUES = DSL.arrayAgg(HAS_ERRORS).as("has_errors_values");

    private static final Field<String[]> RESPONSIBLES_ARRAY = DSL.arrayAgg(ResponsiblesTable.ID).as("responsibles");

    private final SecurityContext securityContext;

    private static final Map<RuleCondition.Source, RuleConditionFormatter> FORMATTERS;

    static {
        ImmutableMap.Builder<RuleCondition.Source, RuleConditionFormatter> builder = ImmutableMap.builder();
        FORMATTERS = builder
            .put(RuleCondition.Source.BROWSER_SITES, UrlFormatter.withPath)
            .put(RuleCondition.Source.SITES, UrlFormatter.withPath)
            .put(RuleCondition.Source.YANDEX_REFERRER, UrlFormatter.withPath)
            .put(RuleCondition.Source.SEARCH_RESULTS_HOSTS, UrlFormatter.withoutPath)
            .put(RuleCondition.Source.PUBLIC_SITES, UrlFormatter.withPath)

            .put(RuleCondition.Source.WORDS, WordRuleFormatter.instance)
            .put(RuleCondition.Source.METRICA_TITLES, WordRuleFormatter.instance)
            .put(RuleCondition.Source.SEARCH_REQUESTS, WordRuleFormatter.instance)
            .put(RuleCondition.Source.BROWSER_TITLES, WordRuleFormatter.instance)
            .put(RuleCondition.Source.PUBLIC_WORDS, WordRuleFormatter.instance)

            .put(RuleCondition.Source.MUSIC_LIKES, IntWithCommentFormatter.instance)
            .put(RuleCondition.Source.CATALOGIA, IntWithCommentFormatter.instance)

            .put(RuleCondition.Source.METRICA_COUNTERS_AND_GOALS, MetricaCounterFormatter.instance)
            .put(RuleCondition.Source.APPS, AppFormatter.instance)
            .build();
    }

    public RulesConditionsTable(Configuration configuration, SecurityContext securityContext) {
        super(configuration, RuleCondition.class);
        this.securityContext = securityContext;
    }

    public static Field[] aggregateFields() {
        return new Field[]{
                RULE_IDS,
                SOURCES,
                STATES,
                VALUES,
                REVISIONS,
                CREATED_TIMESTAMPS,
                MODIFIED_TIMESTAMPS,
                HAS_ERRORS_VALUES,
        };
    }

    @Override
    public Select<Record> selectQuery() {
        return dsl.selectFrom(TABLE);
    }

    public Select<Record> selectQuery(String ruleId) {
        return dsl.selectFrom(TABLE)
                .where(RULE_ID.eq(ruleId));
    }

    public Select<Record> selectQuery(String ruleId, RuleCondition.Source source) {
        return dsl.selectFrom(TABLE)
                .where(RULE_ID.eq(ruleId).and(SOURCE.eq(source.toString())));
    }

    public Select<Record> selectQuery(String ruleId, RuleCondition.Source source, RuleCondition.State state) {
        return dsl.selectFrom(TABLE)
                .where(RULE_ID.eq(ruleId).and(SOURCE.eq(source.toString())).and(STATE.eq(state.toString())));
    }

    public Select<Record> selectUnapprovedByRuleIdQuery(String ruleId) {
        return dsl.selectFrom(TABLE)
                .where(RULE_ID.eq(ruleId).and(STATE.eq(RuleCondition.State.NEED_APPROVE.toString())));
    }

    public Select<Record> selectBySourceQuery(RuleCondition.Source source) {
        return dsl.selectFrom(TABLE)
                .where(SOURCE.eq(source.toString()));
    }

    public Select<Record> selectAccessibleQuery(String ruleId, RuleCondition.Source source) {
        return selectAccessibleQuery(ruleId, source, Optional.empty());
    }

    public Select<Record> selectAccessibleQuery(String ruleId, RuleCondition.Source source, Optional<RuleCondition.State> state) {
        return selectQuery(ruleId, source, state, RulesAcl.isAccessibleBy(securityContext));
    }

    public Select<Record> selectByRevisionAccessibleQuery(Long revision) {
        return dsl.selectFrom(TABLE.join(RulesTable.TABLE).on(RULE_ID.equal(RulesTable.ID)))
                .where(REVISION.eq(revision).and(RulesAcl.isAccessibleBy(securityContext)));
    }


    public Table<Record> joinedTable() {
        return TABLE
                .leftJoin(RulesTable.TABLE)
                .on(RULE_ID.equal(RulesTable.ID))
                .leftJoin(SegmentExportsTable.TABLE)
                .on(RULE_ID.eq(SegmentExportsTable.RULE_ID))
                .leftJoin(ResponsiblesTable.TABLE)
                .on(ResponsiblesTable.SEGMENT_ID.eq(SegmentExportsTable.SEGMENT_ID));

    }

    private Field[] selectedFields() {
        return new Field[]{
                RULE_ID,
                SOURCE,
                STATE,
                VALUE,
                REVISION,
                CREATED,
                MODIFIED,
                HAS_ERRORS,
                RESPONSIBLES_ARRAY,
        };
    }

    private Select<Record> selectQuery(String ruleId, RuleCondition.Source source, Optional<RuleCondition.State> state, Condition aclCondition) {
        var condition = RULE_ID.eq(ruleId)
                .and(SOURCE.eq(source.toString()))
                .and(aclCondition);

        if (state.isPresent()) {
            condition = condition.and(STATE.eq(state.get().toString()));
        }

        return dsl.select(selectedFields())
                .from(joinedTable())
                .where(condition)
                .groupBy(
                        RULE_ID,
                        SOURCE,
                        STATE,
                        VALUE,
                        REVISION,
                        CREATED,
                        MODIFIED,
                        HAS_ERRORS
                );
    }

    private Condition conditionsAcl() {
        return RulesAcl
                .isAuthor(securityContext)
                .or(RulesAcl.isResponsible(securityContext))
                .or(RulesAcl.isAdmin(securityContext));
    }

    public Select<Record> selectModifiableQuery(String ruleId, RuleCondition.Source source) {
        return selectQuery(ruleId, source, Optional.empty(), conditionsAcl());
    }

    public Select<Record> selectModifiableQuery(String ruleId, RuleCondition.Source source, RuleCondition.State state) {
        return selectQuery(ruleId, source, Optional.of(state), conditionsAcl());
    }

    public Select<Record> selectApprovableQuery(String ruleId, RuleCondition.Source source, RuleCondition.State state) {
        return selectQuery(ruleId, source, Optional.of(state), RulesAcl.isApprovableBy(securityContext));
    }

    @Override
    protected RuleCondition read(Record record) {
        return readCondition(record);
    }

    public static RuleCondition readCondition(Record record) {
        var source = RuleCondition.Source.valueOf(record.get(SOURCE));
        var formattedLines = RuleFormatUtils.parseValues(getFormatter(source), record.get(VALUE));
        var fullValues = RuleFormatUtils.getFullValue(formattedLines);

        return RuleCondition.newBuilder()
                .setRuleId(record.get(RULE_ID))
                .setSource(source)
                .setState(RuleCondition.State.valueOf(record.get(STATE)))
                .addAllValues(RuleFormatUtils.formatValues(formattedLines))
                .addAllFullValues(fullValues)
                .setRevision(record.get(REVISION))
                .setTimestamps(Timestamps.newBuilder()
                        .setCreated(record.get(CREATED))
                        .setModified(record.get(MODIFIED)))
                .setHasErrors(record.get(HAS_ERRORS))
                .build();
    }

    protected static Stream<RuleCondition> readAggregated(Record record) {
        String[] ruleIds = record.get(RULE_IDS);
        String[] sources = record.get(SOURCES);
        String[] states = record.get(STATES);
        String[] values = record.get(VALUES);
        Long[] revisions = record.get(REVISIONS);
        Long[] createdTimestamps = record.get(CREATED_TIMESTAMPS);
        Long[] modifiedTimestamps = record.get(MODIFIED_TIMESTAMPS);
        Boolean[] hasErrorsValues = record.get(HAS_ERRORS_VALUES);

        return IntStream.range(0, ruleIds.length)
                .filter(index -> Objects.nonNull(ruleIds[index]))
                .mapToObj(index -> {
                    var source = RuleCondition.Source.valueOf(sources[index]);
                    var formattedLines = RuleFormatUtils.parseValues(getFormatter(source), values[index]);
                    var fullValues = RuleFormatUtils.getFullValue(formattedLines);

                    return RuleCondition.newBuilder()
                                    .setRuleId(ruleIds[index])
                                    .setSource(source)
                                    .setState(RuleCondition.State.valueOf(states[index]))
                                    .addAllValues(RuleFormatUtils.formatValues(formattedLines))
                                    .addAllFullValues(fullValues)
                                    .setRevision(revisions[index])
                                    .setTimestamps(Timestamps.newBuilder()
                                            .setCreated(createdTimestamps[index])
                                            .setModified(modifiedTimestamps[index]))
                                    .setHasErrors(hasErrorsValues[index])
                                    .build();
                }).distinct();
    }

    public Query insertQuery(String ruleId, RuleCondition.Source source, RuleCondition.State state,
                             Long timestamp, List<String> values, Boolean hasErrors) {
        return dsl.insertInto(TABLE)
                .set(RULE_ID, ruleId)
                .set(SOURCE, source.toString())
                .set(STATE, state.toString())
                .set(VALUE, RuleFormatUtils.serializeValues(values))
                .set(CREATED, timestamp)
                .set(MODIFIED, timestamp)
                .set(HAS_ERRORS, hasErrors);
    }

    public Update<Record> approveQuery(String ruleId, RuleCondition.Source source, Long timestamp) {
        return dsl.update(TABLE)
                .set(STATE, RuleCondition.State.APPROVED.toString())
                .set(MODIFIED, timestamp)
                .where(RULE_ID.eq(ruleId).and(SOURCE.eq(source.toString()))
                        .and(STATE.eq(RuleCondition.State.NEED_APPROVE.toString())));
    }

    public Update<Record> updateValueQuery(String ruleId, RuleCondition.Source source, RuleCondition.State state,
            Long timestamp, List<String> values, Boolean hasErrors)
    {
        return dsl.update(TABLE)
                .set(REVISION, REVISION_SEQ.nextval())
                .set(VALUE, RuleFormatUtils.serializeValues(values))
                .set(MODIFIED, timestamp)
                .set(HAS_ERRORS, hasErrors)
                .where(RULE_ID.eq(ruleId).and(SOURCE.eq(source.toString())).and(STATE.eq(state.toString())));
    }

    public DeleteConditionStep<Record> deleteQuery(String ruleId, RuleCondition.Source source) {
        return dsl.deleteFrom(TABLE)
                .where(RULE_ID.eq(ruleId).and(SOURCE.eq(source.toString())));
    }

    public DeleteConditionStep<Record> deleteQuery(String ruleId, RuleCondition.Source source,
            RuleCondition.State state)
    {
        return dsl.deleteFrom(TABLE)
                .where(RULE_ID.eq(ruleId).and(SOURCE.eq(source.toString())).and(STATE.eq(state.toString())));
    }

    public DeleteConditionStep<Record> deleteByRuleIdQuery(String ruleId) {
        return dsl.deleteFrom(TABLE)
                .where(RULE_ID.eq(ruleId));
    }

    public static RuleConditionFormatter getFormatter(RuleCondition.Source source) {
        return FORMATTERS.getOrDefault(source, IdentityFormatter.instance);
    }
}
