package ru.yandex.crypta.lab.tables;

import java.util.stream.Stream;

import javax.ws.rs.core.SecurityContext;

import org.jooq.Configuration;
import org.jooq.DeleteConditionStep;
import org.jooq.Field;
import org.jooq.Query;
import org.jooq.Record;
import org.jooq.SelectHavingConditionStep;
import org.jooq.SelectHavingStep;
import org.jooq.Table;
import org.jooq.TableOnConditionStep;
import org.jooq.Update;
import org.jooq.impl.DSL;

import ru.yandex.crypta.common.data.GenericTable;
import ru.yandex.crypta.lab.proto.Rule;
import ru.yandex.crypta.lab.proto.Timestamps;


public class RulesTable extends GenericTable<Rule> {

    public static final Table<Record> TABLE = DSL.table("api_constructor_rules");

    public static final Field<String> ID = DSL.field(DSL.name(TABLE.getName(), "id"), String.class);
    public static final Field<String> AUTHOR = DSL.field(DSL.name(TABLE.getName(), "author"), String.class);
    public static final Field<String> NAME = DSL.field(DSL.name(TABLE.getName(), "name"), String.class);
    public static final Field<Long> DAYS = DSL.field(DSL.name(TABLE.getName(), "days"), Long.class);
    public static final Field<Long> MIN_DAYS = DSL.field(DSL.name(TABLE.getName(), "min_days"), Long.class);
    public static final Field<String> ISSUE_ID = DSL.field(DSL.name(TABLE.getName(), "issue_id"), String.class);
    public static final Field<String> ISSUE_KEY = DSL.field(DSL.name(TABLE.getName(), "issue_key"), String.class);
    private static final Field<Long> CREATED = DSL.field(DSL.name(TABLE.getName(), "created"), Long.class);
    private static final Field<Long> MODIFIED = DSL.field(DSL.name(TABLE.getName(), "modified"), Long.class);

    private final SecurityContext securityContext;

    public RulesTable(Configuration configuration, SecurityContext securityContext) {
        super(configuration, Rule.class);
        this.securityContext = securityContext;
    }

    private static Field[] selectFields() {
        Field[] tableFields = {
                ID,
                AUTHOR,
                NAME,
                DAYS,
                MIN_DAYS,
                CREATED,
                MODIFIED,
                ISSUE_ID,
                ISSUE_KEY
        };
        Field[] conditionsAggregates = RulesConditionsTable.aggregateFields();
        return Stream.concat(
                Stream.of(tableFields),
                Stream.of(conditionsAggregates)
        ).toArray(Field[]::new);
    }

    private TableOnConditionStep<Record> joinedTable() {
        return TABLE
                .leftOuterJoin(RulesConditionsTable.TABLE)
                .on(ID.eq(RulesConditionsTable.RULE_ID));
    }

    @Override
    public SelectHavingStep<Record> selectQuery() {
        return dsl.select(selectFields())
                .from(joinedTable())
                .groupBy(ID);
    }


    public SelectHavingConditionStep<Record> selectByIdQuery(String id) {
        return selectQuery()
                .having(ID.eq(id));
    }

    public SelectHavingConditionStep<Record> selectByIdModifiableQuery(String id) {
        return selectQuery()
                .having(ID.eq(id).and(RulesAcl.isModifiableBy(securityContext)));
    }

    public SelectHavingConditionStep<Record> selectByIdAccessibleQuery(String id) {
        return selectQuery()
                .having(ID.eq(id).and(RulesAcl.isAccessibleBy(securityContext)));
    }

    @Override
    protected Rule read(Record record) {
        var builder = Rule.newBuilder()
                .setId(record.get(ID))
                .setAuthor(record.get(AUTHOR))
                .setName(record.get(NAME))
                .setDays(record.get(DAYS))
                .setMinDays(record.get(MIN_DAYS))
                .setTimestamps(Timestamps.newBuilder()
                        .setCreated(record.get(CREATED))
                        .setModified(record.get(MODIFIED)))
                .setIssueId(record.get(ISSUE_ID))
                .setIssueKey(record.get(ISSUE_KEY));

        RulesConditionsTable.readAggregated(record).forEachOrdered(builder::addConditions);
        return builder.build();
    }

    public Query insertQuery(Rule rule)
    {
        return dsl.insertInto(TABLE)
                .set(ID, rule.getId())
                .set(AUTHOR, rule.getAuthor())
                .set(NAME, rule.getName())
                .set(DAYS, rule.getDays())
                .set(MIN_DAYS, rule.getMinDays())
                .set(CREATED, rule.getTimestamps().getCreated())
                .set(MODIFIED, rule.getTimestamps().getModified())
                .set(ISSUE_ID, rule.getIssueId())
                .set(ISSUE_KEY, rule.getIssueKey());
    }

    public Update<Record> updateQuery(String id, Rule rule) {
        return dsl.update(TABLE)
                .set(AUTHOR, rule.getAuthor())
                .set(NAME, rule.getName())
                .set(DAYS, rule.getDays())
                .set(MIN_DAYS, rule.getMinDays())
                .set(CREATED, rule.getTimestamps().getCreated())
                .set(MODIFIED, rule.getTimestamps().getModified())
                .set(ISSUE_ID, rule.getIssueId())
                .set(ISSUE_KEY, rule.getIssueKey())
                .where(ID.eq(id));
    }

    public DeleteConditionStep<Record> deleteQuery(String id) {
        return dsl.deleteFrom(TABLE)
                .where(ID.eq(id));
    }
}
