package ru.yandex.crypta.lab.tables;

import javax.ws.rs.core.SecurityContext;

import org.jooq.Condition;
import org.jooq.Configuration;
import org.jooq.Delete;
import org.jooq.Field;
import org.jooq.Query;
import org.jooq.Record;
import org.jooq.Select;
import org.jooq.Table;
import org.jooq.TableOnConditionStep;
import org.jooq.impl.DSL;
import org.jooq.impl.SQLDataType;

import ru.yandex.crypta.common.data.GenericTable;
import ru.yandex.crypta.lab.proto.ESampleViewState;
import ru.yandex.crypta.lab.proto.ESampleViewType;
import ru.yandex.crypta.lab.proto.TSampleView;
import ru.yandex.crypta.lab.proto.TSampleViewOptions;
import ru.yandex.crypta.lab.utils.Proto;

public class SampleViewsTable extends GenericTable<TSampleView> {

    public static final Table<Record> TABLE = DSL.table("api_lab_sample_views");

    public static final Field<String> SAMPLE_ID = DSL.field(DSL.name(TABLE.getName(), "sample_id"), String.class);
    public static final Field<String> ID = DSL.field(DSL.name(TABLE.getName(), "id"), String.class);
    public static final Field<String> PATH = DSL.field(DSL.name(TABLE.getName(), "path"), String.class);
    public static final Field<String> STATE = DSL.field(DSL.name(TABLE.getName(), "state"), String.class);
    public static final Field<String> ERROR = DSL.field(DSL.name(TABLE.getName(), "error"), String.class);
    public static final Field<byte[]> OPTIONS = DSL.field(DSL.name(TABLE.getName(), "options"), SQLDataType.BINARY);
    public static final Field<String> VIEW_TYPE = DSL.field(DSL.name(TABLE.getName(), "type"), String.class);

    private final SecurityContext securityContext;

    public SampleViewsTable(Configuration configuration, SecurityContext securityContext) {
        super(configuration, TSampleView.class);
        this.securityContext = securityContext;
    }

    private static Condition hasId(String sampleId, String viewId) {
        return ID.equal(viewId).and(SAMPLE_ID.equal(sampleId));
    }

    @Override
    protected TSampleView read(Record record) {
        return TSampleView.newBuilder()
                .setSampleID(record.get(SAMPLE_ID))
                .setID(record.get(ID))
                .setPath(record.get(PATH))
                .setState(ESampleViewState.valueOf(record.get(STATE)))
                .setOptions(Proto.readOrDefault(record.get(OPTIONS), TSampleViewOptions.parser(),
                        TSampleViewOptions.getDefaultInstance()))
                .setType(ESampleViewType.valueOf(record.get(VIEW_TYPE)))
                .setError(record.get(ERROR))
                .build();
    }

    @Override
    public Select selectQuery() {
        return dsl.selectFrom(TABLE).where(SamplesAcl.isPublic());
    }

    private TableOnConditionStep<Record> joinedTable() {
        return SamplesTable.TABLE.join(TABLE).on(SamplesTable.ID.equal(SAMPLE_ID));
    }

    public Select<Record> selectBySampleIdQuery(String sampleId) {
        return dsl.selectFrom(joinedTable())
                .where(SAMPLE_ID.equal(sampleId))
                .and(SamplesAcl.isAccessibleDirectlyBy(securityContext));
    }

    public Select<Record> selectByIdQuery(String sampleId, String viewId) {
        return dsl.selectFrom(joinedTable())
                .where(SAMPLE_ID.equal(sampleId).and(ID.equal(viewId)))
                .and(SamplesAcl.isAccessibleDirectlyBy(securityContext));
    }

    public Delete deleteBySampleIdQuery(String sampleId) {
        return dsl.deleteFrom(TABLE)
                .where(SAMPLE_ID.equal(sampleId));
    }

    public Query insertQuery(TSampleView view) {
        return dsl.insertInto(TABLE)
                .set(ID, view.getID())
                .set(SAMPLE_ID, view.getSampleID())
                .set(PATH, view.getPath())
                .set(STATE, view.getState().name())
                .set(OPTIONS, Proto.write(view.getOptions()))
                .set(VIEW_TYPE, view.getType().name())
                .set(ERROR, view.getError());
    }

    public Select<Record> selectOutdatedQuery(Long now) {
        Condition overdue = DSL
                .value(now)
                .minus(SamplesTable.TTL)
                .greaterOrEqual(SamplesTable.MODIFIED);
        Condition notDeleted = STATE
                .notEqual(ESampleViewState.DELETED.name());
        return dsl.selectFrom(joinedTable())
                .where(overdue)
                .and(notDeleted);
    }

    public Query updateStateQuery(String sampleId, String viewId, ESampleViewState state) {
        return dsl.update(TABLE)
                .set(STATE, state.name())
                .where(hasId(sampleId, viewId));
    }

    public Query updateErrorQuery(String sampleId, String viewId, String error) {
        return dsl.update(TABLE)
                .set(ERROR, error)
                .where(hasId(sampleId, viewId));
    }

    public Query updateQuery(TSampleView view) {
        return dsl.update(TABLE)
                .set(STATE, view.getState().name())
                .set(PATH, view.getPath())
                .set(OPTIONS, Proto.write(view.getOptions()))
                .set(VIEW_TYPE, view.getType().name())
                .set(ERROR, view.getError())
                .where(hasId(view.getSampleID(), view.getID()));
    }

    public Query insertOrUpdateQuery(TSampleView view) {
        return dsl.insertInto(TABLE)
                .set(ID, view.getID())
                .set(SAMPLE_ID, view.getSampleID())
                .set(PATH, view.getPath())
                .set(STATE, view.getState().name())
                .set(ERROR, view.getError())
                .set(VIEW_TYPE, view.getType().name())
                .set(OPTIONS, Proto.write(view.getOptions()))
                .onConflict(ID, SAMPLE_ID)
                .doUpdate()
                .set(PATH, view.getPath())
                .set(STATE, view.getState().name())
                .set(ERROR, view.getError())
                .set(VIEW_TYPE, view.getType().name())
                .set(OPTIONS, Proto.write(view.getOptions()));
    }

}
