package ru.yandex.crypta.lab.tables;

import java.util.Optional;

import javax.ws.rs.core.SecurityContext;

import org.jooq.Condition;
import org.jooq.Configuration;
import org.jooq.Delete;
import org.jooq.Field;
import org.jooq.Insert;
import org.jooq.Query;
import org.jooq.Record;
import org.jooq.Select;
import org.jooq.Table;
import org.jooq.Update;
import org.jooq.impl.DSL;

import ru.yandex.crypta.common.data.GenericTable;
import ru.yandex.crypta.lab.proto.AccessLevel;
import ru.yandex.crypta.lab.proto.Sample;
import ru.yandex.crypta.lab.proto.SampleType;

public class SamplesTable extends GenericTable<Sample> {

    public static final Table<Record> TABLE = DSL.table("api_lab_samples");
    public static final Field<String> ID = DSL.field(DSL.name(TABLE.getName(), "id"), String.class);
    public static final Long DEFAULT_TTL = 604_800L;
    public static final Field<String> AUTHOR = DSL.field(DSL.name(TABLE.getName(), "author"), String.class);
    public static final Field<String> ACCESS_LEVEL =
            DSL.field(DSL.name(TABLE.getName(), "access_level"), String.class);
    public static final Field<Long> MODIFIED = DSL.field(DSL.name(TABLE.getName(), "modified"), Long.class);
    public static final Field<Long> TTL = DSL.field(DSL.name(TABLE.getName(), "ttl"), Long.class);
    private static final Field<String> NAME = DSL.field(DSL.name(TABLE.getName(), "name"), String.class);
    private static final Field<String> ID_TYPE = DSL.field(DSL.name(TABLE.getName(), "id_type"), String.class);
    private static final Field<String> ID_KEY = DSL.field(DSL.name(TABLE.getName(), "id_key"), String.class);
    private static final Field<String> DATE_KEY = DSL.field(DSL.name(TABLE.getName(), "date_key"), String.class);
    private static final Field<String> GROUPING_KEY =
            DSL.field(DSL.name(TABLE.getName(), "grouping_key"), String.class);
    private static final Field<Integer> MAX_GROUPS_COUNT = DSL.field(DSL.name(TABLE.getName(), "max_groups_count"), Integer.class);
    private static final Field<Long> CREATED = DSL.field(DSL.name(TABLE.getName(), "created"), Long.class);
    private static final Field<String> TYPE = DSL.field(DSL.name(TABLE.getName(), "type"), String.class);
    private static final Field<String> SIBERIA_USER_SET_ID =
            DSL.field(DSL.name(TABLE.getName(), "cdp_id"), String.class);
    private static final Field<String> STATE = DSL.field(DSL.name(TABLE.getName(), "state"), String.class);

    private final SecurityContext securityContext;

    public SamplesTable(Configuration configuration, SecurityContext securityContext) {
        super(configuration, Sample.class);
        this.securityContext = securityContext;
    }

    private static Condition hasId(String id) {
        return ID.eq(id);
    }

    @Override
    protected Sample read(Record record) {
        Sample.Builder builder = Sample.newBuilder()
                .setId(record.get(ID))
                .setAuthor(record.get(AUTHOR))
                .setName(record.get(NAME))
                .setIdName(record.get(ID_TYPE))
                .setIdKey(record.get(ID_KEY))
                .setDateKey(Optional.ofNullable(record.get(DATE_KEY)).orElse(""))
                .setGroupingKey(record.get(GROUPING_KEY))
                .setTtl(record.get(TTL))
                .setAccessLevel(AccessLevel.valueOf(record.get(ACCESS_LEVEL)))
                .setMaxGroupsCount(record.get(MAX_GROUPS_COUNT))
                .setType(SampleType.valueOf(record.get(TYPE)))
                .setSiberiaUserSetId(Optional.ofNullable(record.get(SIBERIA_USER_SET_ID)).orElse(""))
                .setState(Optional.ofNullable(record.get(STATE)).orElse(""))
                .putAllUserSetIdToGroupingKeyValue(SubsamplesTable.getSubsampleIdsBuilders(record));

        builder.getTimestampsBuilder()
                .setModified(record.get(MODIFIED))
                .setCreated(record.get(CREATED));

        return builder.build();
    }

    private Table<Record> samplesJoined() {
        return TABLE
                .leftJoin(SubsamplesTable.TABLE)
                .on(SubsamplesTable.SAMPLE_ID.eq(ID));
    }

    @Override
    public Select<Record> selectQuery() {
        return dsl.select(aggregatedFields())
                .from(samplesJoined())
                .where(SamplesAcl.isAdmin(securityContext))
                .groupBy(ID);
    }

    private Field[] aggregatedFields() {
        Field[] fields = {
                ID,
                AUTHOR,
                ACCESS_LEVEL,
                CREATED,
                MODIFIED,
                TTL,
                NAME,
                ID_TYPE,
                ID_KEY,
                DATE_KEY,
                GROUPING_KEY,
                MAX_GROUPS_COUNT,
                TYPE,
                SIBERIA_USER_SET_ID,
                STATE,
                SubsamplesTable.SUBSAMPLE_IDS
        };

        return fields;
    }

    public Select<Record> selectByIdQuery(String id) {
        return dsl.select(aggregatedFields())
                .from(samplesJoined())
                .where(hasId(id))
                .and(SamplesAcl.isAccessibleDirectlyBy(securityContext))
                .groupBy(ID);
    }

    public Delete<Record> deleteByIdQuery(String id) {
        return dsl.deleteFrom(TABLE)
                .where(hasId(id))
                .and(SamplesAcl.isModifiableBy(securityContext));
    }

    public Insert<Record> insertQuery(Sample sample) {
        return dsl.insertInto(TABLE)
                .set(ID, sample.getId())
                .set(AUTHOR, sample.getAuthor())
                .set(ACCESS_LEVEL, sample.getAccessLevel().name())
                .set(NAME, sample.getName())
                .set(ID_TYPE, sample.getIdName())
                .set(ID_KEY, sample.getIdKey())
                .set(DATE_KEY, sample.getDateKey())
                .set(GROUPING_KEY, sample.getGroupingKey())
                .set(MAX_GROUPS_COUNT, sample.getMaxGroupsCount())
                .set(MODIFIED, sample.getTimestamps().getModified())
                .set(CREATED, sample.getTimestamps().getCreated())
                .set(TYPE, sample.getType().name())
                .set(TTL, sample.getTtl())
                .set(STATE, sample.getState())
                .set(SIBERIA_USER_SET_ID, sample.getSiberiaUserSetId());
    }

    public Update<Record> updateQuery(Sample sample) {
        return dsl.update(TABLE)
                .set(AUTHOR, sample.getAuthor())
                .set(ACCESS_LEVEL, sample.getAccessLevel().name())
                .set(NAME, sample.getName())
                .set(ID_TYPE, sample.getIdName())
                .set(ID_KEY, sample.getIdKey())
                .set(DATE_KEY, sample.getDateKey())
                .set(GROUPING_KEY, sample.getGroupingKey())
                .set(MAX_GROUPS_COUNT, sample.getMaxGroupsCount())
                .set(MODIFIED, sample.getTimestamps().getModified())
                .set(CREATED, sample.getTimestamps().getCreated())
                .set(TTL, sample.getTtl())
                .set(TYPE, sample.getType().name())
                .set(SIBERIA_USER_SET_ID, sample.getSiberiaUserSetId())
                .set(STATE, sample.getState())
                .where(hasId(sample.getId()).and(SamplesAcl.isModifiableBy(securityContext)));
    }

    public Query updateSampleStateQuery(String id, String state) {
        return dsl.update(TABLE)
                .set(STATE, state)
                .where(hasId(id).and(SamplesAcl.isModifiableBy(securityContext)));
    }

    public Select<Record> selectAccessible() {
        return dsl.select(aggregatedFields())
                .from(samplesJoined())
                .where(SamplesAcl.isListedFor(securityContext))
                .groupBy(ID);
    }

    public Select<Record> selectOutdatedQuery(Long now) {
        Condition overdue = DSL.value(now).minus(TTL).greaterOrEqual(MODIFIED);

        return dsl.select(aggregatedFields()).from(samplesJoined()).where(overdue).groupBy(ID);
    }
}
