package ru.yandex.crypta.lab.tables;

import java.time.Duration;
import java.time.Instant;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import javax.ws.rs.core.SecurityContext;

import org.jooq.Configuration;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Query;
import org.jooq.Record;
import org.jooq.Record1;
import org.jooq.Select;
import org.jooq.Table;
import org.jooq.Update;
import org.jooq.impl.DSL;

import ru.yandex.crypta.common.data.GenericTable;
import ru.yandex.crypta.lab.formatters.ExpressionFormatter;
import ru.yandex.crypta.lab.formatters.RuleConditionFormatter;
import ru.yandex.crypta.lab.formatters.RuleFormatUtils;
import ru.yandex.crypta.lab.proto.Coverage;
import ru.yandex.crypta.lab.proto.Segment;
import ru.yandex.crypta.lab.proto.SegmentAttributes;

public class SegmentExportsTable extends GenericTable<Segment.Export> {

    public static final Table<Record> TABLE = DSL.table("api_segment_exports");

    public static final Field<String> ID =
            DSL.field(DSL.name(TABLE.getName(), "id"), String.class);
    public static final Field<String> SEGMENT_ID =
            DSL.field(DSL.name(TABLE.getName(), "segment_id"), String.class);
    public static final Field<Long> EXPORT_KEYWORD_ID =
            DSL.field(DSL.name(TABLE.getName(), "export_keyword_id"), Long.class);
    public static final Field<Long> EXPORT_SEGMENT_ID =
            DSL.field(DSL.name(TABLE.getName(), "export_segment_id"), Long.class);
    public static final Field<String> TYPE =
            DSL.field(DSL.name(TABLE.getName(), "type"), String.class);
    public static final Field<String> STATE =
            DSL.field(DSL.name(TABLE.getName(), "state"), String.class);
    private static final Field<Long> COVERAGE_PROFILES_VALUE =
            DSL.field(DSL.name(TABLE.getName(), "coverage_profiles_value"), Long.class);
    private static final Field<Long> COVERAGE_PROFILES_TIMESTAMP =
            DSL.field(DSL.name(TABLE.getName(), "coverage_profiles_timestamp"), Long.class);
    private static final Field<Long> COVERAGE_BIGB_VALUE =
            DSL.field(DSL.name(TABLE.getName(), "coverage_bigb_value"), Long.class);
    private static final Field<Long> COVERAGE_BIGB_TIMESTAMP =
            DSL.field(DSL.name(TABLE.getName(), "coverage_bigb_timestamp"), Long.class);
    private static final Field<String> EXPRESSION =
            DSL.field(DSL.name(TABLE.getName(), "expression"), String.class);
    private static final Field<String> LAL =
            DSL.field(DSL.name(TABLE.getName(), "lal"), String.class);
    public static final Field<String> RULE_ID = DSL.field(DSL.name(TABLE.getName(), "rule_id"), String.class);
    private static final Field<Boolean> RULE_HAS_ERRORS = DSL.field(DSL.name(TABLE.getName(), "rule_has_errors"), Boolean.class);
    private static final Field<Boolean> EXPORT_TO_BB = DSL.field(DSL.name(TABLE.getName(), "export_to_bb"), Boolean.class);
    private static final Field<Long> NEXT_ACTIVITY_CHECK_TIMESTAMP =
            DSL.field(DSL.name(TABLE.getName(), "next_activity_check_timestamp"), Long.class);
    private static final Field<String> EXPORT_TYPE_ID =
            DSL.field(DSL.name(TABLE.getName(), "export_type_id"), String.class);

    public static final Field<String[]> IDS = DSL.arrayAgg(ID).as("ids");
    private static final Field<String[]> TYPES = DSL.arrayAgg(TYPE).as("types");
    public static final Field<String[]> STATES = DSL.arrayAgg(STATE).as("states");
    private static final Field<Long[]> EXPORT_KEYWORD_IDS = DSL.arrayAgg(EXPORT_KEYWORD_ID).as("export_keyword_ids");
    private static final Field<Long[]> EXPORT_SEGMENT_IDS = DSL.arrayAgg(EXPORT_SEGMENT_ID).as("export_segment_ids");
    public static final Field<Long[]> COVERAGE_PROFILES_VALUES =
            DSL.arrayAgg(COVERAGE_PROFILES_VALUE).as("coverage_profiles_values");
    public static final Field<Long[]> COVERAGE_PROFILES_TIMESTAMPS =
            DSL.arrayAgg(COVERAGE_PROFILES_TIMESTAMP).as("coverage_profiles_timestamps");
    private static final Field<Long[]> COVERAGE_BIGB_VALUES =
            DSL.arrayAgg(COVERAGE_BIGB_VALUE).as("coverage_bigb_values");
    private static final Field<Long[]> COVERAGE_BIGB_TIMESTAMPS =
            DSL.arrayAgg(COVERAGE_BIGB_TIMESTAMP).as("coverage_bigb_timestamps");
    private static final Field<String[]> EXPRESSIONS = DSL.arrayAgg(EXPRESSION).as("expressions");
    private static final Field<String[]> LALS = DSL.arrayAgg(LAL).as("lal");
    private static final Field<String[]> RULE_IDS = DSL.arrayAgg(RULE_ID).as("rule_ids");
    private static final Field<Boolean[]> RULE_HAS_ERRORS_VALUES = DSL.arrayAgg(RULE_HAS_ERRORS).as("rule_has_errors_values");
    private static final Field<String[]> TAGS = DSL.arrayAgg(SegmentExportsTagsTable.AGG_TAGS).as("tags");
    private static final Field<Long[]> NEXT_ACTIVITY_CHECK_TIMESTAMPS =
            DSL.arrayAgg(NEXT_ACTIVITY_CHECK_TIMESTAMP).as("next_activity_check_timestamps");
    private static final Field<String[]> EXPORT_TYPE_IDS =
            DSL.arrayAgg(EXPORT_TYPE_ID).as("export_type_ids");

    private final RuleConditionFormatter formatter;

    private final SecurityContext securityContext;

    public SegmentExportsTable(Configuration configuration, SecurityContext securityContext) {
        super(configuration, Segment.Export.class);
        this.securityContext = securityContext;
        this.formatter = new ExpressionFormatter(ids -> getUnknownExports(dsl, ids));
    }

    public static Field[] aggregateFields() {
        return new Field[]{
                IDS,
                TYPES,
                STATES,
                EXPORT_KEYWORD_IDS,
                EXPORT_SEGMENT_IDS,
                COVERAGE_PROFILES_VALUES,
                COVERAGE_PROFILES_TIMESTAMPS,
                COVERAGE_BIGB_VALUES,
                COVERAGE_BIGB_TIMESTAMPS,
                EXPRESSIONS,
                LALS,
                RULE_IDS,
                RULE_HAS_ERRORS_VALUES,
                NEXT_ACTIVITY_CHECK_TIMESTAMPS,
                EXPORT_TYPE_IDS,
        };
    }

    public static Stream<Segment.Export> readAggregated(Record record, DSLContext dsl) {
        String[] ids = record.get(IDS);
        String[] types = record.get(TYPES);
        String[] states = record.get(STATES);
        Long[] keywordIds = record.get(EXPORT_KEYWORD_IDS);
        Long[] segmentIds = record.get(EXPORT_SEGMENT_IDS);
        Long[] coverageProfileValues = record.get(COVERAGE_PROFILES_VALUES);
        Long[] coverageProfileTimestamps = record.get(COVERAGE_PROFILES_TIMESTAMPS);
        Long[] coverageBigbValues = record.get(COVERAGE_BIGB_VALUES);
        Long[] coverageBigbTimestamps = record.get(COVERAGE_BIGB_TIMESTAMPS);
        String[] expressions = record.get(EXPRESSIONS);
        String[] lals = record.get(LALS);
        String[] ruleIds = record.get(RULE_IDS);
        Boolean[] ruleHasErrors = record.get(RULE_HAS_ERRORS_VALUES);
        String[] tags = record.get(TAGS);
        Long[] nextActivityCheckTimestamps = record.get(NEXT_ACTIVITY_CHECK_TIMESTAMPS);
        String[] exportTypeIds = record.get(EXPORT_TYPE_IDS);

        var formatter = new ExpressionFormatter(exports -> getUnknownExports(dsl, exports));

        return IntStream.range(0, ids.length)
                .filter(index -> Objects.nonNull(ids[index]))
                .mapToObj(index -> {
                    Segment.Export.Builder builder = Segment.Export.newBuilder()
                            .setId(ids[index])
                            .setType(Segment.Export.Type.valueOf(types[index]))
                            .setState(Segment.Export.State.valueOf(states[index]))
                            .setNextActivityCheckTimestamp(nextActivityCheckTimestamps[index])
                            .setKeywordId(keywordIds[index])
                            .setSegmentId(segmentIds[index])
                            .addAllTags(SegmentExportsTagsTable.readAggregated(tags[index]))
                            .setExportTypeId(exportTypeIds[index]);

                    if (coverageProfileValues[index] != null) {
                        Coverage.Builder profilesCoverage = builder.getCoveragesBuilder().getProfilesBuilder();
                        profilesCoverage.setValue(coverageProfileValues[index]);
                        if (coverageProfileTimestamps[index] != null) {
                            profilesCoverage.setTimestamp(coverageProfileTimestamps[index]);
                        }
                    }

                    if (coverageBigbValues[index] != null) {
                        Coverage.Builder bigbCoverage = builder.getCoveragesBuilder().getBigbBuilder();
                        bigbCoverage.setValue(coverageBigbValues[index]);
                        if (coverageBigbTimestamps[index] != null) {
                            bigbCoverage.setTimestamp(coverageBigbTimestamps[index]);
                        }
                    }

                    if (expressions[index] != null) {
                        var formattedLines = RuleFormatUtils.parseValues(formatter, expressions[index]);
                        var fullValues = RuleFormatUtils.getFullValue(formattedLines);
                        builder.addAllExpressions(RuleFormatUtils.formatValues(formattedLines));
                        builder.addAllFullExpressions(fullValues);
                        builder.setHasExpressionErrors(RuleFormatUtils.hasErrors(fullValues));
                    }

                    if (lals[index] != null) {
                        builder.setLal(lals[index]);
                    }

                    if (ruleIds[index] != null) {
                        builder.setRuleId(ruleIds[index]);
                        builder.setRuleHasErrors(ruleHasErrors[index]);
                    }

                    return builder.build();
                }).distinct();
    }

    public static Stream<SegmentAttributes.ExportState> readAggregatedStatus(Record record) {
        String[] ids = record.get(IDS);
        String[] states = record.get(STATES);
        Long[] coverageProfilesValues = record.get(COVERAGE_PROFILES_VALUES);
        Long[] coverageProfilesTimestamps = record.get(COVERAGE_PROFILES_TIMESTAMPS);

        return IntStream.range(0, ids.length)
                .filter(index -> Objects.nonNull(ids[index]))
                .mapToObj(index -> {
                    SegmentAttributes.ExportState.Builder builder = SegmentAttributes.ExportState.newBuilder()
                            .setId(ids[index])
                            .setState(Segment.Export.State.valueOf(states[index]));

                    if (coverageProfilesValues[index] != null) {
                        builder.setProfilesCoverage(coverageProfilesValues[index]);

                        Instant now = Instant.now();
                        Duration limit = Duration.ofDays(5);
                        if (coverageProfilesTimestamps[index] != null) {
                            Instant coverageTime = Instant.ofEpochSecond(coverageProfilesTimestamps[index]);
                            Duration coverageDuration = Duration.between(coverageTime, now);
                            boolean overdue = coverageDuration.compareTo(limit) > 0;

                            builder.setOverdue(overdue);
                        }
                    }

                    return builder.build();
                }).distinct();
    }

    @Override
    protected Segment.Export read(Record record) {
        Segment.Export.Builder builder = Segment.Export.newBuilder()
                .setId(record.get(ID))
                .setType(Segment.Export.Type.valueOf(record.get(TYPE)))
                .setState(Segment.Export.State.valueOf(record.get(STATE)))
                .setNextActivityCheckTimestamp(record.get(NEXT_ACTIVITY_CHECK_TIMESTAMP))
                .setKeywordId(record.get(EXPORT_KEYWORD_ID))
                .setSegmentId(record.get(EXPORT_SEGMENT_ID))
                .setExportToBb(record.get(EXPORT_TO_BB))
                .setRuleHasErrors(record.get(RULE_HAS_ERRORS))
                .addAllTags(SegmentExportsTagsTable.readAggregated(record))
                .setExportTypeId(record.get(EXPORT_TYPE_ID));

        Coverage.Builder profilesCoverage = builder.getCoveragesBuilder()
                .getProfilesBuilder();
        if (record.get(COVERAGE_PROFILES_VALUE) != null) {
            profilesCoverage.setValue(record.get(COVERAGE_PROFILES_VALUE));
            if (record.get(COVERAGE_PROFILES_TIMESTAMP) != null) {
                profilesCoverage.setTimestamp(record.get(COVERAGE_PROFILES_TIMESTAMP));
            }
        }
        Coverage.Builder bigbCoverage = builder.getCoveragesBuilder()
                .getBigbBuilder();
        if (record.get(COVERAGE_BIGB_VALUE) != null) {
            bigbCoverage.setValue(record.get(COVERAGE_BIGB_VALUE));
            if (record.get(COVERAGE_BIGB_TIMESTAMP) != null) {
                bigbCoverage.setTimestamp(record.get(COVERAGE_BIGB_TIMESTAMP));
            }
        }

        if (record.get(EXPRESSION) != null) {
            var formattedLines = RuleFormatUtils.parseValues(formatter, record.get(EXPRESSION));
            var fullValues = RuleFormatUtils.getFullValue(formattedLines);
            builder.addAllExpressions(RuleFormatUtils.formatValues(formattedLines));
            builder.addAllFullExpressions(fullValues);
            builder.setHasExpressionErrors(RuleFormatUtils.hasErrors(fullValues));
        }

        if (record.get(LAL) != null) {
            builder.setLal(record.get(LAL));
        }

        if (record.get(RULE_ID) != null) {
            builder.setRuleId(record.get(RULE_ID));
        }

        return builder.build();
    }

    @Override
    public Select<Record> selectQuery() {
        return dsl.selectFrom(TABLE
                .leftJoin(SegmentExportsTagsTable.aggregateTable(dsl)).on(ID.eq(SegmentExportsTagsTable.EXPORT_ID)));
    }

    public Select<Record> selectQuery(String id) {
        return dsl.selectFrom(TABLE
                .leftJoin(SegmentExportsTagsTable.aggregateTable(dsl)).on(ID.eq(SegmentExportsTagsTable.EXPORT_ID)))
                .where(ID.eq(id));
    }

    public static Set<String> getUnknownExports(DSLContext dsl, Set<String> ids) {
        // TODO(CRYPTA-14866) revert when slowdown problems are resolved
        // var result = new HashSet<>(ids);
        // dsl.selectFrom(TABLE).where(ID.in(ids)).fetch(ID).forEach(result::remove);
        // return result;

        return new HashSet<>();
    }

    public Select<Record> selectModifiableQuery(String id) {
        return dsl.selectFrom(TABLE
                .join(SegmentsTable.TABLE).on(SEGMENT_ID.equal(SegmentsTable.ID))
                .leftJoin(SegmentExportsTagsTable.aggregateTable(dsl)).on(ID.eq(SegmentExportsTagsTable.EXPORT_ID))
                .leftJoin(ResponsiblesTable.TABLE).on(SEGMENT_ID.eq(ResponsiblesTable.SEGMENT_ID)))
                .where(ID.eq(id).and(SegmentsAcl.isModifiableBy(securityContext)));
    }

    public Select<Record> selectQueryWithRuleId() {
        return dsl.selectFrom(TABLE
                .leftJoin(SegmentExportsTagsTable.aggregateTable(dsl)).on(ID.eq(SegmentExportsTagsTable.EXPORT_ID)))
                .where(RULE_ID.isNotNull());
    }

    public Select<Record> selectQueryWithExportIdInExpressions(String exportId) {
        return dsl.selectFrom(TABLE
                .leftJoin(SegmentExportsTagsTable.aggregateTable(dsl)).on(ID.eq(SegmentExportsTagsTable.EXPORT_ID)))
                .where(EXPRESSION.contains(exportId));
    }

    public Select<Record1<Long>> selectLastSegmentIdQuery(List<Long> keywordIds) {
        return dsl.select(DSL.max(EXPORT_SEGMENT_ID))
                .from(TABLE)
                .where(EXPORT_KEYWORD_ID.in(keywordIds));
    }

    public Select<Record> selectNotExportedToBigBQuery() {
        return dsl.selectFrom(TABLE
                .leftJoin(SegmentExportsTagsTable.aggregateTable(dsl)).on(ID.eq(SegmentExportsTagsTable.EXPORT_ID)))
                .where(EXPORT_TO_BB.eq(false));
    }

    public Query insertQuery(String segmentId, Segment.Export.Builder export) {
        return dsl.insertInto(TABLE)
                .set(SEGMENT_ID, segmentId)
                .set(ID, export.getId())
                .set(TYPE, export.getType().toString())
                .set(STATE, export.getState().toString())
                .set(NEXT_ACTIVITY_CHECK_TIMESTAMP, export.getNextActivityCheckTimestamp())
                .set(EXPORT_KEYWORD_ID, export.getKeywordId())
                .set(EXPORT_SEGMENT_ID, export.getSegmentId())
                .set(COVERAGE_PROFILES_VALUE, export.getCoverages().getProfiles().getValue())
                .set(COVERAGE_PROFILES_TIMESTAMP, export.getCoverages().getProfiles().getTimestamp())
                .set(COVERAGE_BIGB_VALUE, export.getCoverages().getBigb().getValue())
                .set(COVERAGE_BIGB_TIMESTAMP, export.getCoverages().getBigb().getTimestamp())
                .set(EXPORT_TYPE_ID, export.getExportTypeId());
    }

    private Query setExportToBigbQuery(String exportId, boolean flag) {
        return dsl.update(TABLE)
                .set(EXPORT_TO_BB, flag)
                .where(ID.eq(exportId));
    }

    public Update<Record> updateRuleError(String ruleId, Boolean flag) {
        return dsl.update(TABLE)
                .set(RULE_HAS_ERRORS, flag)
                .where(RULE_ID.eq(ruleId));
    }

    public Query enableExportToBigbQuery(String exportId) {
        return setExportToBigbQuery(exportId, true);
    }

    public Query disableExportToBigbQuery(String exportId) {
        return setExportToBigbQuery(exportId, false);
    }

    public Query updateSegmentIdQuery(String exportId, String segmentId) {
        return dsl.update(TABLE)
                .set(SEGMENT_ID, segmentId)
                .where(ID.eq(exportId));
    }

    public Query updateQuery(String exportId, Segment.Export.Builder export) {
        return dsl.update(TABLE)
                .set(TYPE, export.getType().toString())
                .set(STATE, export.getState().toString())
                .set(NEXT_ACTIVITY_CHECK_TIMESTAMP, export.getNextActivityCheckTimestamp())
                .set(EXPORT_KEYWORD_ID, export.getKeywordId())
                .set(EXPORT_SEGMENT_ID, export.getSegmentId())
                .set(COVERAGE_PROFILES_VALUE, export.getCoverages().getProfiles().getValue())
                .set(COVERAGE_PROFILES_TIMESTAMP, export.getCoverages().getProfiles().getTimestamp())
                .set(COVERAGE_BIGB_VALUE, export.getCoverages().getBigb().getValue())
                .set(COVERAGE_BIGB_TIMESTAMP, export.getCoverages().getBigb().getTimestamp())
                .set(EXPORT_TYPE_ID, export.getExportTypeId())
                .where(ID.eq(exportId));
    }

    public Query updateExpressionQuery(String exportId, List<String> expression) {
        return dsl.update(TABLE)
                .set(EXPRESSION, RuleFormatUtils.serializeValues(expression))
                .where(ID.eq(exportId));
    }

    public Query deleteExpressionQuery(String exportId) {
        return dsl.update(TABLE)
                .set(EXPRESSION, (String) null)
                .where(ID.eq(exportId));
    }

    public Query updateLalQuery(String exportId, String lal) {
        return dsl.update(TABLE)
                .set(LAL, lal)
                .where(ID.eq(exportId));
    }

    public Query deleteLalQuery(String exportId) {
        return dsl.update(TABLE)
                .set(LAL, (String) null)
                .where(ID.eq(exportId));
    }

    public Query updateRuleIdQuery(String exportId, String ruleId) {
        return dsl.update(TABLE)
                .set(RULE_ID, ruleId)
                .where(ID.eq(exportId));
    }

    public Query deleteRuleIdQuery(String exportId) {
        return dsl.update(TABLE)
                .set(RULE_ID, (String) null)
                .where(ID.eq(exportId));
    }

    public Query updateExportStateQuery(String id, Segment.Export.State state) {
        return dsl.update(TABLE)
                .set(STATE, state.toString())
                .where(ID.eq(id));
    }

    public Query updateExportNextActivityCheckTs(String id, Long timestamp) {
        return dsl.update(TABLE)
                .set(NEXT_ACTIVITY_CHECK_TIMESTAMP, timestamp)
                .where(ID.eq(id));
    }

    public Query deleteQuery(String id) {
        return dsl.deleteFrom(TABLE)
                .where(ID.eq(id));
    }

    public Query deleteBySegmentId(String segmentId) {
        return dsl.deleteFrom(TABLE).where(SEGMENT_ID.equal(segmentId));
    }
}
