package ru.yandex.crypta.lab.tables;

import java.util.Arrays;
import java.util.List;

import org.jooq.Configuration;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.InsertValuesStep2;
import org.jooq.Query;
import org.jooq.Record;
import org.jooq.Record2;
import org.jooq.Select;
import org.jooq.SelectHavingStep;
import org.jooq.Table;
import org.jooq.impl.DSL;

import ru.yandex.crypta.common.data.GenericTable;
import ru.yandex.crypta.lab.proto.Segment;

public class SegmentExportsTagsTable extends GenericTable<Segment.Tags> {

    public static final Table<Record> TABLE = DSL.table("api_segment_export_tags");

    private static final String DELIMITER = ";";

    public static final Field<String> SEGMENT_EXPORT_ID =
            DSL.field(DSL.name(TABLE.getName(), "segment_export_id"), String.class);
    public static final Field<String> TAG =
            DSL.field(DSL.name(TABLE.getName(), "tag"), String.class);

    public static final Field<String> TAGS = DSL.groupConcat(TAG, DELIMITER).as("tags");

    private static final String AGGREGATED_TAGS_TABLE_NAME = "aggTags";

    public static final Field<String> AGG_TAGS = DSL.field(DSL.name(AGGREGATED_TAGS_TABLE_NAME, "tags"), String.class);
    public static final Field<String> EXPORT_ID = DSL.field(DSL.name(AGGREGATED_TAGS_TABLE_NAME, "segment_export_id"), String.class);
    public static final Field<String> AGG_TAGS_EMPTY_DEFAULT = DSL.when(AGG_TAGS.isNotNull(), AGG_TAGS).otherwise("");

    public static Table aggregateTable(DSLContext dsl) {
        return dsl.select(SegmentExportsTagsTable.SEGMENT_EXPORT_ID, SegmentExportsTagsTable.TAGS).
                from(SegmentExportsTagsTable.TABLE).groupBy(SegmentExportsTagsTable.SEGMENT_EXPORT_ID).asTable(AGGREGATED_TAGS_TABLE_NAME);
    }

    public static Field[] aggregateFields() {
        return new Field[]{
                DSL.arrayAgg(AGG_TAGS_EMPTY_DEFAULT).as("tags"),
        };
    }

    public SegmentExportsTagsTable(Configuration configuration) {
        super(configuration, Segment.Tags.class);
    }

    public static List<String> readAggregated(Record record) {
        if (record.get(TAGS) != null) {
            return readAggregated(record.get(TAGS));
        } else {
            return Arrays.asList();
        }
    }

    public static List<String> readAggregated(String tags) {
        if (tags.equals("")) {
            return Arrays.asList();
        } else {
            return Arrays.asList(tags.split(DELIMITER));
        }
    }

    @Override
    protected Segment.Tags read(Record record) {
        return Segment.Tags.newBuilder()
                .setExportId(record.get(SEGMENT_EXPORT_ID))
                .addAllTags(Arrays.asList(record.get(TAGS).split(DELIMITER)))
                .build();
    }

    @Override
    public Select selectQuery() {
        return dsl.select(SEGMENT_EXPORT_ID, TAGS).from(TABLE).groupBy(SEGMENT_EXPORT_ID);
    }

    public SelectHavingStep<Record2<String, String>> selectQuery(String id) {
        return dsl.select(SEGMENT_EXPORT_ID, TAGS).from(TABLE)
                .where(SEGMENT_EXPORT_ID.eq(id)).groupBy(SEGMENT_EXPORT_ID);
    }

    public Query insertQuery(Segment.Tags.Builder tags) {
        InsertValuesStep2<Record, String, String> ins = dsl.insertInto(TABLE, SEGMENT_EXPORT_ID, TAG);
        String exportID = tags.getExportId();
        for (String tag : tags.getTagsList()){
            ins = ins.values(exportID, tag);
        }
        return ins;
    }

    public Query deleteQuery(Segment.Tags.Builder tags) {
        return dsl.deleteFrom(TABLE)
                .where(TAG.in(tags.getTagsList()).and(SEGMENT_EXPORT_ID.eq(tags.getExportId())));
    }

    public Query insertQuery(String id, String tag) {
        return dsl.insertInto(TABLE, SEGMENT_EXPORT_ID, TAG).values(id, tag);
    }

    public Query deleteQuery(String id, String tag) {
        return dsl.deleteFrom(TABLE)
                .where(TAG.eq(tag).and(SEGMENT_EXPORT_ID.eq(id)));
    }
}
