package ru.yandex.crypta.lab.tables;

import java.util.List;
import java.util.stream.Collectors;

import org.jooq.Batch;
import org.jooq.Configuration;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Select;
import org.jooq.SelectConditionStep;
import org.jooq.Table;
import org.jooq.Truncate;
import org.jooq.impl.DSL;

import ru.yandex.crypta.common.data.GenericTable;
import ru.yandex.crypta.lab.proto.SegmentIdPair;

public class SegmentIdsTable extends GenericTable<SegmentIdPair> {
    public static final Table<Record> TABLE = DSL.table("api_segment_id_to_user_set_id");
    public static final Field<String> SEGMENT_LAB_ID = DSL.field(
            DSL.name(TABLE.getName(), "segment_lab_id"),
            String.class
    );
    public static final Field<String> USER_SET_ID = DSL.field(
            DSL.name(TABLE.getName(), "user_set_id"),
            String.class
    );

    public SegmentIdsTable(Configuration configuration) {
        super(configuration, SegmentIdPair.class);
    }

    @Override
    public SegmentIdPair read(Record record) {
        return readSegmentIdPair(record);
    }

    public static SegmentIdPair readSegmentIdPair(Record record) {
        return SegmentIdPair.newBuilder()
                .setSegmentLabId(record.get(SEGMENT_LAB_ID))
                .setUserSetId(record.get(USER_SET_ID))
                .build();
    }

    public SelectConditionStep<Record> selectBySegmentIdQuery(String id) {
        return dsl.selectFrom(TABLE).where(SEGMENT_LAB_ID.eq(id));
    }

    @Override
    public Select<Record> selectQuery() {
        return dsl.selectFrom(TABLE);
    }

    public Truncate<Record> truncateTableQuery() {
        return dsl.truncate(TABLE);
    }

    public Batch batchInsertQuery(List<SegmentIdPair> segmentIdPairs) {
        return dsl.batch(
                segmentIdPairs.stream().map(item -> dsl.insertInto(TABLE)
                        .set(SEGMENT_LAB_ID, item.getSegmentLabId())
                        .set(USER_SET_ID, item.getUserSetId())).collect(Collectors.toList())
        );
    }
}
