package ru.yandex.crypta.lab.tables;

import java.time.Instant;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.ws.rs.core.SecurityContext;

import org.jooq.ArrayAggOrderByStep;
import org.jooq.Condition;
import org.jooq.Configuration;
import org.jooq.DeleteConditionStep;
import org.jooq.Field;
import org.jooq.Query;
import org.jooq.Record;
import org.jooq.Record1;
import org.jooq.Record2;
import org.jooq.Record3;
import org.jooq.Row2;
import org.jooq.Select;
import org.jooq.SelectConditionStep;
import org.jooq.SelectHavingConditionStep;
import org.jooq.SelectHavingStep;
import org.jooq.Table;
import org.jooq.TableOnConditionStep;
import org.jooq.Update;
import org.jooq.impl.DSL;

import ru.yandex.crypta.common.data.GenericTable;
import ru.yandex.crypta.lab.I18Utils;
import ru.yandex.crypta.lab.proto.Segment;
import ru.yandex.crypta.lab.proto.SegmentAttributes;
import ru.yandex.crypta.lab.proto.SegmentConditions;
import ru.yandex.crypta.lab.proto.SegmentGroup;
import ru.yandex.crypta.lab.proto.Timestamps;

public class SegmentsTable extends GenericTable<Segment> {

    public static final Table<Record> TABLE = DSL.table("api_segments");
    public static final Field<String> ID = DSL.field(DSL.name(TABLE.getName(), "id"), String.class);
    public static final Field<String> NAME_RU = DSL.field(DSL.name(TABLE.getName(), "name_ru"), String.class);
    public static final Field<String> NAME_EN = DSL.field(DSL.name(TABLE.getName(), "name_en"), String.class);
    public static final Field<String> NAME =
            DSL.field(DSL.name(TABLE.getName(), "name"), String.class);

    public static final Field<String> DESCRIPTION_EN =
            DSL.field(DSL.name(TABLE.getName(), "description_en"), String.class);
    public static final Field<String> DESCRIPTION_RU =
            DSL.field(DSL.name(TABLE.getName(), "description_ru"), String.class);

    public static final Field<String> TANKER_NAME_KEY =
            DSL.field(DSL.name(TABLE.getName(), "tanker_name_key"), String.class);
    public static final Field<String> TANKER_DESCRIPTION_KEY =
            DSL.field(DSL.name(TABLE.getName(), "tanker_description_key"), String.class);

    public static final Field<String> SCOPE = DSL.field(DSL.name(TABLE.getName(), "scope"), String.class);
    public static final Field<String> TYPE = DSL.field(DSL.name(TABLE.getName(), "type"), String.class);
    private static final Field<String> DESCRIPTION =
            DSL.field(DSL.name(TABLE.getName(), "description"), String.class);
    private static final Field<String> STATE = DSL.field(DSL.name(TABLE.getName(), "state"), String.class);
    private static final Field<String> TICKET = DSL.field(DSL.name(TABLE.getName(), "ticket"), String.class);
    private static final Field<Long> PRIORITY = DSL.field(DSL.name(TABLE.getName(), "priority"), Long.class);
    private static final Field<Long> CREATED = DSL.field(DSL.name(TABLE.getName(), "created"), Long.class);
    private static final Field<Long> MODIFIED = DSL.field(DSL.name(TABLE.getName(), "modified"), Long.class);
    private static final Field<String> PARENT_ID = DSL.field(DSL.name(TABLE.getName(), "parent_id"), String.class);
    private static final String DELIMITER = ";";
    public static final Field<String> AUTHOR = DSL.field(DSL.name(TABLE.getName(), "author"), String.class);
    private static final Field<Record2<String, Long>[]> RESPONSIBLES_ARRAY =
            distinctPacked(ResponsiblesTable.ID, ResponsiblesTable.MODIFIED).as("responsibles");
    private static final Field<Record2<String, Long>[]> STAKEHOLDERS_ARRAY =
            distinctPacked(StakeholdersTable.ID, StakeholdersTable.MODIFIED).as("stakeholders");

    private final SecurityContext securityContext;

    public SegmentsTable(Configuration configuration, SecurityContext securityContext) {
        super(configuration, Segment.class);
        this.securityContext = securityContext;
    }

    private static ArrayAggOrderByStep<Record2<String, Long>[]> distinctPacked(Field<String> myField,
            Field<Long> order)
    {
        Row2<String, Long> row = DSL.row(myField, order);
        return DSL.arrayAggDistinct(DSL.rowField(row));
    }

    private static Condition typeIsNotGroup() {
        return TYPE.notEqual(Segment.Type.GROUP.toString());
    }

    private static Condition typeIsGroup() {
        return TYPE.equal(Segment.Type.GROUP.toString());
    }

    public Segment readSegment(Record record) {
        String segmentId = record.get(ID);
        Segment.Exports.Builder exports = Segment.Exports.newBuilder();
        Timestamps.Builder timestamps = Timestamps
                .newBuilder()
                .setCreated(record.get(CREATED))
                .setModified(record.get(MODIFIED));
        Segment.Builder prototype = Segment.newBuilder()
                .setId(segmentId)
                .setAuthor(record.get(AUTHOR))
                .setScope(Segment.Scope.valueOf(record.get(SCOPE)))
                .addAllTickets(Arrays.asList(record.get(TICKET).split(DELIMITER)))
                .setType(Segment.Type.valueOf(record.get(TYPE)))
                .setState(Segment.State.valueOf(record.get(STATE)))
                .setPriority(record.get(PRIORITY))
                .setTimestamps(timestamps)
                .setExports(exports)
                .setParentId(record.get(PARENT_ID))
                .setName(I18Utils.pack(I18Utils.string(record.get(NAME_EN), record.get(NAME_RU))))
                .setDescription(I18Utils.pack(
                        I18Utils.string(record.get(DESCRIPTION_EN), record.get(DESCRIPTION_RU)))
                )
                .setTankerNameKey(Optional.ofNullable(record.get(TANKER_NAME_KEY)).orElse(""))
                .setTankerDescriptionKey(Optional.ofNullable(record.get(TANKER_DESCRIPTION_KEY)).orElse(""));

        Comparator<Record2<String, Long>> orderBySecond = Comparator.comparing(Record2::value2);

        Stream.of(record.get(RESPONSIBLES_ARRAY))
                .filter(each -> Objects.nonNull(each.value1()))
                .distinct()
                .sorted(orderBySecond)
                .map(Record2::value1)
                .forEach(prototype::addResponsibles);
        Stream.of(record.get(STAKEHOLDERS_ARRAY))
                .filter(each -> Objects.nonNull(each.value1()))
                .distinct()
                .sorted(orderBySecond)
                .map(Record2::value1)
                .forEach(prototype::addStakeholders);
        ModelSegmentRelationsTable.readAggregated(record).forEachOrdered(prototype::addModels);
        SegmentExportsTable.readAggregated(record, dsl).forEachOrdered(prototype.getExportsBuilder()::addExports);
        return prototype.build();
    }

    public static SegmentAttributes readSegmentAttributes(Record record) {
        SegmentAttributes.Builder prototype = SegmentAttributes.newBuilder()
                .setId(record.get(ID))
                .setType(Segment.Type.valueOf(record.get(TYPE)))
                .setState(Segment.State.valueOf(record.get(STATE)))
                .setPriority(record.get(PRIORITY))
                .setParentId(record.get(PARENT_ID))
                .setName(I18Utils.pack(I18Utils.string(record.get(NAME_EN), record.get(NAME_RU))));

        SegmentExportsTable.readAggregatedStatus(record).forEachOrdered(prototype::addExportStates);

        return prototype.build();
    }

    public static SegmentGroup readGroup(Record record) {
        return SegmentGroup
                .newBuilder()
                .setId(record.get(ID))
                .setAuthor(record.get(AUTHOR))
                .setParentId(record.get(PARENT_ID))
                .setName(I18Utils.pack(I18Utils.string(record.get(NAME_EN), record.get(NAME_RU))))
                .setDescription(I18Utils.pack(
                        I18Utils.string(record.get(DESCRIPTION_EN), record.get(DESCRIPTION_RU))))
                .setPriority(record.get(PRIORITY))
                .setTankerNameKey(Optional.ofNullable(record.get(TANKER_NAME_KEY)).orElse(""))
                .setTankerDescriptionKey(Optional.ofNullable(record.get(TANKER_DESCRIPTION_KEY)).orElse(""))
                .build();
    }

    @Override
    public Segment read(Record record) {
        return readSegment(record);
    }

    public Query updateQuery(Segment segment) {
        I18Utils.I18String name = I18Utils.unpack(segment.getName());
        I18Utils.I18String description = I18Utils.unpack(segment.getDescription());

        return dsl.update(TABLE)
                .set(TICKET, segment.getTicketsList().stream().collect(Collectors.joining(DELIMITER)))
                .set(SCOPE, segment.getScope().name())
                .set(TYPE, segment.getType().name())
                .set(NAME_EN, name.getEn())
                .set(NAME_RU, name.getRu())
                .set(DESCRIPTION_EN, description.getEn())
                .set(DESCRIPTION_RU, description.getRu())
                .set(TANKER_NAME_KEY, segment.getTankerNameKey())
                .set(TANKER_DESCRIPTION_KEY, segment.getTankerDescriptionKey())
                .set(CREATED, segment.getTimestamps().getCreated())
                .set(MODIFIED, Instant.now().getEpochSecond())
                .set(PARENT_ID, segment.getParentId())
                .set(STATE, segment.getState().toString())
                .where(ID.eq(segment.getId()));
    }

    public Query updateSegmentGroupQuery(SegmentGroup group) {
        I18Utils.I18String name = I18Utils.unpack(group.getName());
        I18Utils.I18String description = I18Utils.unpack(group.getDescription());

        return dsl.update(TABLE)
                .set(NAME_EN, name.getEn())
                .set(NAME_RU, name.getRu())
                .set(DESCRIPTION_EN, description.getEn())
                .set(DESCRIPTION_RU, description.getRu())
                .set(TANKER_NAME_KEY, group.getTankerNameKey())
                .set(TANKER_DESCRIPTION_KEY, group.getTankerDescriptionKey())
                .set(PARENT_ID, group.getParentId())
                .where(ID.eq(group.getId()));
    }

    public Query updateParentIdQuery(String segmentId, String newParentId) {
        return dsl.update(TABLE)
                .set(PARENT_ID, newParentId)
                .where(ID.eq(segmentId));
    }

    public Query insertQuery(Segment segment) {
        return genericInsertQuery(TABLE, segment);
    }

    private Query genericInsertQuery(Table<Record> table, Segment segment) {
        I18Utils.I18String name = I18Utils.unpack(segment.getName());
        I18Utils.I18String description = I18Utils.unpack(segment.getDescription());

        return dsl.insertInto(table)
                .set(ID, segment.getId())
                .set(AUTHOR, segment.getAuthor())
                .set(TICKET, segment.getTicketsList().stream().collect(Collectors.joining(DELIMITER)))
                .set(SCOPE, segment.getScope().name())
                .set(TYPE, segment.getType().name())
                .set(PRIORITY, segment.getPriority())
                .set(NAME_EN, name.getEn())
                .set(NAME_RU, name.getRu())
                .set(DESCRIPTION_EN, description.getEn())
                .set(DESCRIPTION_RU, description.getRu())
                .set(TANKER_NAME_KEY, segment.getTankerNameKey())
                .set(TANKER_DESCRIPTION_KEY, segment.getTankerDescriptionKey())
                .set(CREATED, segment.getTimestamps().getCreated())
                .set(MODIFIED, segment.getTimestamps().getModified())
                .set(PARENT_ID, segment.getParentId())
                .set(STATE, segment.getState().toString());
    }

    private TableOnConditionStep<Record> joinedTable() {
        return TABLE
                .leftJoin(SegmentExportsTable.TABLE)
                .on(ID.eq(SegmentExportsTable.SEGMENT_ID))
                .leftJoin(ResponsiblesTable.TABLE)
                .on(ID.eq(ResponsiblesTable.SEGMENT_ID))
                .leftJoin(StakeholdersTable.TABLE)
                .on(ID.eq(StakeholdersTable.SEGMENT_ID))
                .leftJoin(ModelSegmentRelationsTable.TABLE)
                .on(ID.eq(ModelSegmentRelationsTable.SEGMENT_ID))
                .leftJoin(SegmentExportsTagsTable.aggregateTable(dsl))
                .on(SegmentExportsTable.ID.eq(SegmentExportsTagsTable.EXPORT_ID));
    }

    private TableOnConditionStep<Record> joinedForConditionsTable() {
        return TABLE
                .leftJoin(SegmentExportsTable.TABLE)
                .on(ID.eq(SegmentExportsTable.SEGMENT_ID))
                .leftJoin(ResponsiblesTable.TABLE)
                .on(ID.eq(ResponsiblesTable.SEGMENT_ID))
                .leftJoin(StakeholdersTable.TABLE)
                .on(ID.eq(StakeholdersTable.SEGMENT_ID))
                .leftJoin(SegmentExportsTagsTable.TABLE)
                .on(SegmentExportsTable.ID.eq(SegmentExportsTagsTable.SEGMENT_EXPORT_ID));
    }

    private TableOnConditionStep<Record> joinedForAttributesTable() {
        return TABLE
                .leftJoin(SegmentExportsTable.TABLE)
                .on(ID.eq(SegmentExportsTable.SEGMENT_ID));
    }

    @Override
    public SelectHavingConditionStep<Record> selectQuery() {
        return dsl.select(selectFields())
                .from(joinedTable())
                .groupBy(ID).having(typeIsNotGroup());
    }

    public Select<Record> selectAllQuery() {
        return dsl.select(selectFields())
                .from(joinedTable())
                .groupBy(ID);
    }

    public SelectHavingStep<Record> selectAttributesQuery() {
        return dsl.select(attributeFields())
                .from(joinedForAttributesTable())
                .where(typeIsNotGroup())
                .groupBy(ID);

    }

    public SelectHavingStep<Record> selectByExportIdQuery(String exportId) {
        return dsl.select(attributeFields())
                .from(joinedForAttributesTable())
                .where(typeIsNotGroup()).and(SegmentExportsTable.ID.eq(exportId))
                .groupBy(ID);
    }

    public SelectHavingStep<Record> selectByIdsFuzzyQuery(List<String> searchItems) {
        var query = dsl.select(attributeFields())
                .from(joinedForAttributesTable())
                .where();

        for (var searchItem: searchItems) {
            query = query.and(
                    SegmentExportsTable.ID.containsIgnoreCase(searchItem)
                    .or(SegmentExportsTable.RULE_ID.containsIgnoreCase(searchItem))
                    .or(SegmentExportsTable.EXPORT_KEYWORD_ID.cast(String.class).containsIgnoreCase(searchItem))
                    .or(SegmentExportsTable.EXPORT_SEGMENT_ID.cast(String.class).containsIgnoreCase(searchItem))
                    .or(SegmentsTable.ID.containsIgnoreCase(searchItem))
                    .or(SegmentsTable.NAME_EN.containsIgnoreCase(searchItem))
                    .or(SegmentsTable.NAME_RU.containsIgnoreCase(searchItem))
                    .or(SegmentsTable.TICKET.containsIgnoreCase(searchItem))
            );
        }

        return query.groupBy(ID);
    }

    private static Condition isOfAuthor(SegmentConditions conditions) {
        if (conditions.getEmptyAuthor()) {
            return AUTHOR.equal("");
        } else if (conditions.getAuthor().equals("")) {
            return DSL.noCondition();
        }

        return AUTHOR.equal(conditions.getAuthor());
    }

    private static Condition isOfStakeholder(SegmentConditions conditions) {
        if (conditions.getEmptyStakeholder()) {
            return StakeholdersTable.ID.eq("");
        } else if (conditions.getStakeholder().equals("")) {
            return DSL.noCondition();
        }

        return StakeholdersTable.ID.eq(conditions.getStakeholder());
    }

    private static Condition isOfShowDeleted(SegmentConditions conditions) {
        if (!conditions.getShowDeleted()) {
            return STATE.notEqual("DELETED");
        }

        return DSL.noCondition();
    }

    private static Condition isOfTags(SegmentConditions conditions) {
        if (0 == conditions.getTagsCount()) {
            return DSL.noCondition();
        }

        return SegmentExportsTagsTable.TAG.in(Arrays.asList(conditions.getTagsList().toArray()));
    }

    private static Condition isOfKeywordId(SegmentConditions conditions) {
        long keywordId = (long) conditions.getExportKeywordId();

        if (keywordId == 0L) {
            return DSL.noCondition();
        }

        return SegmentExportsTable.EXPORT_KEYWORD_ID.equal(keywordId);
    }

    private static Condition isOfSegmentId(SegmentConditions conditions) {
        long segmentId = (long) conditions.getExportSegmentId();

        if (segmentId == 0L) {
            return DSL.noCondition();
        }

        return SegmentExportsTable.EXPORT_SEGMENT_ID.equal(segmentId);
    }

    private static Condition isOfTicket(SegmentConditions conditions) {
        String ticket = conditions.getTicket();

        if (Objects.equals(ticket, "")) {
            return DSL.noCondition();
        }

        return SegmentsTable.TICKET.containsIgnoreCase(ticket);
    }

    public SelectHavingStep<Record> selectAttributesWithConditionsQuery(SegmentConditions conditions) {
        return dsl.select(attributeFields())
                .from(joinedForConditionsTable())
                .where(
                        typeIsNotGroup()
                        .and(isOfAuthor(conditions))
                        .and(isOfStakeholder(conditions))
                        .and(isOfTags(conditions))
                        .and(isOfKeywordId(conditions))
                        .and(isOfSegmentId(conditions))
                        .and(isOfTicket(conditions))
                        .and(isOfShowDeleted(conditions))
                )
                .groupBy(ID);
    }

    public Select<Record> selectByIdAllTypesQuery(String id) {
        return dsl.select(selectFields())
                .from(joinedTable())
                .where(ID.equal(id)).groupBy(ID);
    }

    private SelectConditionStep<Record> selectByIdBaseQuery(String id) {
        return dsl.select(selectFields())
                .from(joinedTable())
                .where(typeIsNotGroup().and(ID.equal(id)));
    }

    public Select<Record> selectByIdQuery(String id) {
        return selectByIdBaseQuery(id).groupBy(ID);
    }

    public Select<Record> selectByIdModifiableQuery(String id) {
        return selectByIdBaseQuery(id)
                .and(SegmentsAcl.isModifiableBy(securityContext))
                .groupBy(ID);
    }

    public DeleteConditionStep<Record> deleteByIdQuery(String id) {
        return dsl.deleteFrom(TABLE).where(ID.equal(id)).and(typeIsNotGroup());
    }

    private Select<Record1<String>> parents() {
        Table<Record> that = TABLE.as("a");
        Table<Record> thatAgain = TABLE.as("b");
        Field<String> thatParent = DSL.field(DSL.name(that.getName(), PARENT_ID.getName()), String.class);
        Field<String> thatAgainId = DSL.field(DSL.name(thatAgain.getName(), ID.getName()), String.class);
        return dsl.select(thatAgainId)
                .from(that)
                .join(thatAgain)
                .on(thatParent.equal(thatAgainId));
    }

    public SelectConditionStep<Record> selectGroupQuery() {
        Condition isGroup = typeIsGroup().or(ID.in(parents()));
        return dsl.selectFrom(TABLE)
                .where(isGroup);
    }

    public SelectConditionStep<Record> selectGroupOnlyQuery() {
        return dsl.selectFrom(TABLE).where(typeIsGroup());
    }

    public SelectConditionStep<Record> selectGroupByIdQuery(String id) {
        Condition isGroup = typeIsGroup().or(ID.in(parents()));
        Condition requiredId = ID.eq(id);
        return dsl.selectFrom(TABLE)
                .where(requiredId.and(isGroup));
    }

    public Query deleteGroupByIdQuery(String id) {
        return dsl.deleteFrom(TABLE).where(ID.equal(id)).and(typeIsGroup());
    }

    private Field[] attributeFields() {
        Field[] fields = {
                ID,
                NAME_EN,
                NAME_RU,
                DESCRIPTION_EN,
                DESCRIPTION_RU,
                TYPE,
                PARENT_ID,
                PRIORITY,
                STATE,
                SegmentExportsTable.IDS,
                SegmentExportsTable.STATES,
                SegmentExportsTable.COVERAGE_PROFILES_VALUES,
                SegmentExportsTable.COVERAGE_PROFILES_TIMESTAMPS
        };

        return fields;
    }

    private Field[] selectFields() {
        Field[] tableFields = {
                ID,
                AUTHOR,
                NAME,
                NAME_EN,
                NAME_RU,
                DESCRIPTION,
                DESCRIPTION_EN,
                DESCRIPTION_RU,
                TANKER_NAME_KEY,
                TANKER_DESCRIPTION_KEY,
                STATE,
                SCOPE,
                TICKET,
                TYPE,
                PRIORITY,
                CREATED,
                MODIFIED,
                PARENT_ID,
                RESPONSIBLES_ARRAY,
                STAKEHOLDERS_ARRAY,
        };
        Field[] modelsAggregates = ModelSegmentRelationsTable.aggregateFields();
        Field[] exportsAggregates = SegmentExportsTable.aggregateFields();
        Field[] exportTagsAggregates = SegmentExportsTagsTable.aggregateFields();
        return Stream.of(
                tableFields,
                modelsAggregates,
                exportsAggregates,
                exportTagsAggregates)
                .map(Stream::of).reduce(Stream.of(), Stream::concat).toArray(Field[]::new);
    }

    public Query insertGroupQuery(SegmentGroup segmentGroup) {
        I18Utils.I18String name = I18Utils.unpack(segmentGroup.getName());
        I18Utils.I18String description = I18Utils.unpack(segmentGroup.getDescription());
        return dsl.insertInto(TABLE)
                .set(ID, segmentGroup.getId())
                .set(AUTHOR, segmentGroup.getAuthor())
                .set(TICKET, "CRYPTR-46")
                .set(NAME_EN, name.getEn())
                .set(NAME_RU, name.getRu())
                .set(DESCRIPTION_EN, description.getEn())
                .set(DESCRIPTION_RU, description.getRu())
                .set(SCOPE, Segment.Scope.EXTERNAL.toString())
                .set(TYPE, Segment.Type.GROUP.toString())
                .set(PRIORITY, segmentGroup.getPriority())
                .set(PARENT_ID, segmentGroup.getParentId());
    }

    public Select<Record> selectExportsToSimpleSegments() {
        return dsl.selectFrom(
                SegmentExportsTable.TABLE
                        .join(TABLE)
                        .on(SegmentExportsTable.SEGMENT_ID.equal(ID))
        );
    }

    public Select<Record3<Long, String, String>> selectNameWithExportId(Long keywordId) {
        Table<Record> segments = TABLE.as("a");
        Table<Record> exports = SegmentExportsTable.TABLE.as("b");
        Field<Long> exportSegmentId =
                DSL.field(DSL.name(exports.getName(), SegmentExportsTable.EXPORT_SEGMENT_ID.getName()), Long.class);
        Field<Long> exportKeywordId =
                DSL.field(DSL.name(exports.getName(), SegmentExportsTable.EXPORT_KEYWORD_ID.getName()), Long.class);
        Field<String> segmentId =
                DSL.field(DSL.name(exports.getName(), SegmentExportsTable.SEGMENT_ID.getName()), String.class);
        Field<String> id = DSL.field(DSL.name(segments.getName(), ID.getName()), String.class);
        Field<String> nameRu = DSL.field(DSL.name(segments.getName(), NAME_RU.getName()), String.class);
        Field<String> nameEn = DSL.field(DSL.name(segments.getName(), NAME_EN.getName()), String.class);
        return dsl.select(exportSegmentId, nameEn, nameRu)
                .from(exports)
                .join(segments)
                .on(segmentId.equal(id).and(exportKeywordId.eq(keywordId)));
    }

    public Update updatePriorityQuery(String id, Long priority) {
        return dsl.update(TABLE)
                .set(PRIORITY, priority)
                .where(ID.equal(id));
    }
}
