package ru.yandex.crypta.lab.tables;

import org.jooq.Configuration;
import org.jooq.DeleteConditionStep;
import org.jooq.Field;
import org.jooq.Query;
import org.jooq.Record;
import org.jooq.Select;
import org.jooq.SelectConditionStep;
import org.jooq.Table;
import org.jooq.impl.DSL;

import ru.yandex.crypta.common.data.GenericTable;

public class StakeholdersTable extends GenericTable<String> {

    public static final Table<Record> TABLE = DSL.table("api_stakeholders");
    public static final Field<String> ID = DSL.field(DSL.name(TABLE.getName(), "id"), String.class);
    public static final Field<String> SEGMENT_ID =
            DSL.field(DSL.name(TABLE.getName(), "segment_id"), SegmentsTable.ID.getType());
    public static final Field<Long> CREATED = DSL.field(DSL.name(TABLE.getName(), "created"), Long.class);
    public static final Field<Long> MODIFIED = DSL.field(DSL.name(TABLE.getName(), "modified"), Long.class);

    public StakeholdersTable(Configuration configuration) {
        super(configuration, String.class);
    }

    @Override
    protected String read(Record record) {
        return record.get(ID);
    }

    public SelectConditionStep<Record> selectBySegmentIdQuery(String id) {
        return dsl.selectFrom(TABLE).where(SEGMENT_ID.eq(id));
    }

    public SelectConditionStep<Record> selectBySegmentIdAndStakeholderQuery(String segmentId, String stakeholder) {
        return dsl.selectFrom(TABLE).where(SEGMENT_ID.eq(segmentId).and(ID.eq(stakeholder)));
    }

    public DeleteConditionStep<Record> deleteBySegmentIdQuery(String id) {
        return dsl.deleteFrom(TABLE).where(SEGMENT_ID.eq(id));
    }

    private Query genericInsertQuery(Table<Record> table, String stakeholder, String segmentId,
            long created, long modified)
    {
        return dsl.insertInto(table)
                .set(ID, stakeholder)
                .set(SEGMENT_ID, segmentId)
                .set(CREATED, created)
                .set(MODIFIED, modified);
    }

    public Query insertQuery(String stakeholder, String segmentId,
            long created, long modified)
    {
        return genericInsertQuery(TABLE, stakeholder, segmentId, created, modified);
    }

    @Override
    public Select<Record> selectQuery() {
        return dsl.selectFrom(TABLE);
    }

    public DeleteConditionStep<Record> deleteStakeholderQuery(String segmentId, String stakeholder) {
        return dsl.deleteFrom(TABLE).where(SEGMENT_ID.eq(segmentId)).and(ID.eq(stakeholder));
    }

}
